"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WorkflowsExecutionEnginePlugin = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _workflows = require("@kbn/workflows");
var _errors = require("@kbn/workflows/common/errors");
var _uuid = require("uuid");
var _graph = require("@kbn/workflows/graph");
var _common = require("../common");
var _connector_executor = require("./connector_executor");
var _url_validator = require("./lib/url_validator");
var _step_execution_repository = require("./repositories/step_execution_repository");
var _workflow_execution_repository = require("./repositories/workflow_execution_repository");
var _nodes_factory = require("./step/nodes_factory");
var _workflow_execution_runtime_manager = require("./workflow_context_manager/workflow_execution_runtime_manager");
var _workflow_execution_state = require("./workflow_context_manager/workflow_execution_state");
var _workflow_event_logger = require("./workflow_event_logger/workflow_event_logger");
var _workflow_execution_loop = require("./workflow_execution_loop");
var _workflow_task_manager = require("./workflow_task_manager/workflow_task_manager");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class WorkflowsExecutionEnginePlugin {
  constructor(initializerContext) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "config", void 0);
    this.logger = initializerContext.logger.get();
    this.config = initializerContext.config.get();
  }
  setup(core, plugins) {
    this.logger.debug('workflows-execution-engine: Setup');
    const logger = this.logger;
    const config = this.config;
    plugins.taskManager.registerTaskDefinitions({
      'workflow:run': {
        title: 'Run Workflow',
        description: 'Executes a workflow immediately',
        timeout: '5m',
        maxAttempts: 1,
        createTaskRunner: ({
          taskInstance,
          fakeRequest
        }) => {
          const taskAbortController = new AbortController();
          return {
            async run() {
              const {
                workflowRunId,
                spaceId
              } = taskInstance.params;
              const [coreStart, pluginsStart] = await core.getStartServices();
              const {
                actions,
                taskManager
              } = pluginsStart;

              // Get ES client from core services (guaranteed to be available at task execution time)
              const esClient = coreStart.elasticsearch.client.asInternalUser;
              const workflowExecutionRepository = new _workflow_execution_repository.WorkflowExecutionRepository(esClient);
              const {
                workflowRuntime,
                workflowExecutionState,
                workflowLogger,
                nodesFactory,
                workflowExecutionGraph,
                clientToUse,
                fakeRequest: fakeRequestFromContainer,
                coreStart: coreStartFromContainer
              } = await createContainer(workflowRunId, spaceId, actions, taskManager, esClient, logger, config, workflowExecutionRepository, fakeRequest,
              // Provided by Task Manager's first-class API key support
              coreStart);
              await workflowRuntime.start();
              await (0, _workflow_execution_loop.workflowExecutionLoop)({
                workflowRuntime,
                workflowExecutionState,
                workflowExecutionRepository,
                workflowLogger,
                nodesFactory,
                workflowExecutionGraph,
                esClient: clientToUse,
                fakeRequest: fakeRequestFromContainer,
                coreStart: coreStartFromContainer,
                taskAbortController
              });
            },
            async cancel() {
              taskAbortController.abort();
            }
          };
        }
      }
    });
    plugins.taskManager.registerTaskDefinitions({
      'workflow:resume': {
        title: 'Resume Workflow',
        description: 'Resumes a paused workflow',
        timeout: '5m',
        maxAttempts: 1,
        createTaskRunner: ({
          taskInstance,
          fakeRequest
        }) => {
          const taskAbortController = new AbortController();
          return {
            async run() {
              const {
                workflowRunId,
                spaceId
              } = taskInstance.params;
              const [coreStart, pluginsStart] = await core.getStartServices();
              const {
                actions,
                taskManager
              } = pluginsStart;

              // Get ES client from core services (guaranteed to be available at task execution time)
              const esClient = coreStart.elasticsearch.client.asInternalUser;
              const workflowExecutionRepository = new _workflow_execution_repository.WorkflowExecutionRepository(esClient);
              const {
                workflowRuntime,
                workflowExecutionState,
                workflowLogger,
                nodesFactory,
                workflowExecutionGraph,
                clientToUse,
                fakeRequest: fakeRequestFromContainer,
                coreStart: coreStartFromContainer
              } = await createContainer(workflowRunId, spaceId, actions, taskManager, esClient, logger, config, workflowExecutionRepository, fakeRequest,
              // Provided by Task Manager's first-class API key support
              coreStart);
              await workflowRuntime.resume();
              await (0, _workflow_execution_loop.workflowExecutionLoop)({
                workflowRuntime,
                workflowExecutionState,
                workflowExecutionRepository,
                workflowLogger,
                nodesFactory,
                workflowExecutionGraph,
                esClient: clientToUse,
                fakeRequest: fakeRequestFromContainer,
                coreStart: coreStartFromContainer,
                taskAbortController
              });
            },
            async cancel() {
              taskAbortController.abort();
            }
          };
        }
      }
    });
    return {};
  }
  start(core, plugins) {
    this.logger.debug('workflows-execution-engine: Start');
    const executeWorkflow = async (workflow, context, request) => {
      const workflowCreatedAt = new Date();

      // Get ES client and create repository for this execution
      const esClient = core.elasticsearch.client.asInternalUser;
      const workflowExecutionRepository = new _workflow_execution_repository.WorkflowExecutionRepository(esClient);
      const triggeredBy = context.triggeredBy || 'manual'; // 'manual' or 'scheduled'
      const workflowExecution = {
        id: (0, _uuid.v4)(),
        spaceId: context.spaceId,
        workflowId: workflow.id,
        isTestRun: workflow.isTestRun,
        workflowDefinition: workflow.definition,
        yaml: workflow.yaml,
        context,
        status: _workflows.ExecutionStatus.PENDING,
        createdAt: workflowCreatedAt.toISOString(),
        createdBy: context.createdBy || '',
        // TODO: set if available
        triggeredBy // <-- new field for scheduled workflows
      };
      await workflowExecutionRepository.createWorkflowExecution(workflowExecution);

      // AUTO-DETECT: Check if we're already running in a Task Manager context
      const isRunningInTaskManager = triggeredBy === 'scheduled' || context.source === 'task-manager' || (request === null || request === void 0 ? void 0 : request.isFakeRequest) === true;
      if (isRunningInTaskManager) {
        // We're already in a task - execute directly without scheduling another task
        this.logger.info(`Executing workflow directly (already in Task Manager context): ${workflow.id}`);
        const {
          workflowRuntime,
          workflowExecutionState,
          workflowLogger,
          nodesFactory,
          workflowExecutionGraph,
          fakeRequest,
          clientToUse,
          coreStart
        } = await createContainer(workflowExecution.id, workflowExecution.spaceId, plugins.actions, plugins.taskManager, esClient, this.logger, this.config, workflowExecutionRepository, request,
        // Pass the fakeRequest for user context
        core);
        await workflowRuntime.start();
        await (0, _workflow_execution_loop.workflowExecutionLoop)({
          workflowRuntime,
          workflowExecutionState,
          workflowExecutionRepository,
          workflowLogger,
          nodesFactory,
          workflowExecutionGraph,
          esClient: clientToUse,
          fakeRequest,
          coreStart,
          taskAbortController: new AbortController() // TODO: We need to think how to pass this properly from outer task
        });
      } else {
        // Normal manual execution - schedule a task
        const taskInstance = {
          id: `workflow:${workflowExecution.id}:${context.triggeredBy}`,
          taskType: 'workflow:run',
          params: {
            workflowRunId: workflowExecution.id,
            spaceId: workflowExecution.spaceId
          },
          state: {
            lastRunAt: null,
            lastRunStatus: null,
            lastRunError: null
          },
          scope: ['workflows'],
          enabled: true
        };

        // Use Task Manager's first-class API key support by passing the request
        // Task Manager will automatically create and manage the API key
        if (request) {
          // Debug: Log the user info from the original request
          this.logger.info(`Scheduling workflow task with user context for workflow ${workflow.id}`);
          await plugins.taskManager.schedule(taskInstance, {
            request
          });
        } else {
          this.logger.info(`Scheduling workflow task without user context`);
          await plugins.taskManager.schedule(taskInstance);
        }
      }
      return {
        workflowExecutionId: workflowExecution.id
      };
    };
    const executeWorkflowStep = async (workflow, stepId, contextOverride) => {
      const workflowCreatedAt = new Date();

      // Get ES client and create repository for this execution
      const esClient = core.elasticsearch.client.asInternalUser;
      const workflowExecutionRepository = new _workflow_execution_repository.WorkflowExecutionRepository(esClient);
      const context = {
        contextOverride
      };
      const triggeredBy = context.triggeredBy || 'manual'; // 'manual' or 'scheduled'
      const workflowExecution = {
        id: (0, _uuid.v4)(),
        spaceId: workflow.spaceId,
        stepId,
        workflowId: workflow.id,
        isTestRun: workflow.isTestRun,
        workflowDefinition: workflow.definition,
        yaml: workflow.yaml,
        context,
        status: _workflows.ExecutionStatus.PENDING,
        createdAt: workflowCreatedAt.toISOString(),
        createdBy: context.createdBy || '',
        // TODO: set if available
        triggeredBy // <-- new field for scheduled workflows
      };
      await workflowExecutionRepository.createWorkflowExecution(workflowExecution);
      const taskInstance = {
        id: `workflow:${workflowExecution.id}:${context.triggeredBy}`,
        taskType: 'workflow:run',
        params: {
          workflowRunId: workflowExecution.id,
          spaceId: workflowExecution.spaceId
        },
        state: {
          lastRunAt: null,
          lastRunStatus: null,
          lastRunError: null
        },
        scope: ['workflows'],
        enabled: true
      };
      await plugins.taskManager.schedule(taskInstance);
      return {
        workflowExecutionId: workflowExecution.id
      };
    };
    const cancelWorkflowExecution = async (workflowExecutionId, spaceId) => {
      const esClient = core.elasticsearch.client.asInternalUser;
      const workflowExecutionRepository = new _workflow_execution_repository.WorkflowExecutionRepository(esClient);
      const workflowExecution = await workflowExecutionRepository.getWorkflowExecutionById(workflowExecutionId, spaceId);
      if (!workflowExecution) {
        throw new _errors.WorkflowExecutionNotFoundError(workflowExecutionId);
      }
      if ([_workflows.ExecutionStatus.CANCELLED, _workflows.ExecutionStatus.COMPLETED, _workflows.ExecutionStatus.FAILED].includes(workflowExecution.status)) {
        // Already in a terminal state or being canceled
        return;
      }

      // Request cancellation
      await workflowExecutionRepository.updateWorkflowExecution({
        id: workflowExecution.id,
        cancelRequested: true,
        cancellationReason: 'Cancelled by user',
        cancelledAt: new Date().toISOString(),
        cancelledBy: 'system' // TODO: set user if available
      });
      if ([_workflows.ExecutionStatus.WAITING, _workflows.ExecutionStatus.WAITING_FOR_INPUT].includes(workflowExecution.status)) {
        // TODO: handle WAITING states
        // It should clean up resume tasks, etc
      }
    };
    return {
      executeWorkflow,
      executeWorkflowStep,
      cancelWorkflowExecution
    };
  }
  stop() {}
}
exports.WorkflowsExecutionEnginePlugin = WorkflowsExecutionEnginePlugin;
async function createContainer(workflowRunId, spaceId, actionsPlugin, taskManagerPlugin, esClient, logger, config, workflowExecutionRepository, fakeRequest,
// KibanaRequest from task manager
coreStart) {
  const workflowExecution = await workflowExecutionRepository.getWorkflowExecutionById(workflowRunId, spaceId);
  if (!workflowExecution) {
    throw new Error(`Workflow execution with ID ${workflowRunId} not found`);
  }
  let workflowExecutionGraph = _graph.WorkflowGraph.fromWorkflowDefinition(workflowExecution.workflowDefinition);

  // If the execution is for a specific step, narrow the graph to that step
  if (workflowExecution.stepId) {
    workflowExecutionGraph = workflowExecutionGraph.getStepGraph(workflowExecution.stepId);
  }
  const unsecuredActionsClient = await actionsPlugin.getUnsecuredActionsClient();
  const stepExecutionRepository = new _step_execution_repository.StepExecutionRepository(esClient);
  const connectorExecutor = new _connector_executor.ConnectorExecutor(unsecuredActionsClient);
  const workflowLogger = new _workflow_event_logger.WorkflowEventLogger(esClient, logger, _common.WORKFLOWS_EXECUTION_LOGS_INDEX, {
    workflowId: workflowExecution.workflowId,
    workflowName: workflowExecution.workflowDefinition.name,
    executionId: workflowExecution.id,
    spaceId: workflowExecution.spaceId
  }, {
    enableConsoleLogging: config.logging.console
  });
  const workflowExecutionState = new _workflow_execution_state.WorkflowExecutionState(workflowExecution, workflowExecutionRepository, stepExecutionRepository);

  // Create workflow runtime first (simpler, fewer dependencies)
  const workflowRuntime = new _workflow_execution_runtime_manager.WorkflowExecutionRuntimeManager({
    workflowExecution: workflowExecution,
    workflowExecutionGraph,
    workflowLogger,
    workflowExecutionState
  });

  // Use user-scoped ES client if fakeRequest is available, otherwise fallback to regular client
  let clientToUse = esClient; // fallback
  if (fakeRequest && coreStart) {
    clientToUse = coreStart.elasticsearch.client.asScoped(fakeRequest).asCurrentUser;
  }
  const workflowTaskManager = new _workflow_task_manager.WorkflowTaskManager(taskManagerPlugin);
  const urlValidator = new _url_validator.UrlValidator({
    allowedHosts: config.http.allowedHosts
  });
  const nodesFactory = new _nodes_factory.NodesFactory(connectorExecutor, workflowRuntime, workflowExecutionState, workflowLogger, workflowTaskManager, urlValidator, workflowExecutionGraph);
  return {
    workflowExecutionGraph,
    workflowRuntime,
    workflowExecutionState,
    connectorExecutor,
    workflowLogger,
    taskManagerPlugin,
    workflowExecutionRepository,
    workflowTaskManager,
    nodesFactory,
    fakeRequest,
    clientToUse,
    coreStart
  };
}