"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.USER_AVATAR_MAX_INITIALS = exports.USER_AVATAR_FALLBACK_CODE_POINT = void 0;
exports.getUserAvatarColor = getUserAvatarColor;
exports.getUserAvatarInitials = getUserAvatarInitials;
exports.getUserDisplayLabel = getUserDisplayLabel;
exports.getUserDisplayName = getUserDisplayName;
exports.isValidUserProfileAvatarColor = isValidUserProfileAvatarColor;
var _eui = require("@elastic/eui");
var _chromaJs = _interopRequireDefault(require("chroma-js"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * IMPORTANT:
 *
 * The types in this file have been imported from
 * `x-pack/platform/plugins/shared/security/common/model/user_profile.ts`
 *
 * When making changes please ensure to keep both files in sync.
 */

/**
 * Describes basic properties stored in user profile.
 */

/**
 * Basic user information returned in user profile.
 */

const USER_AVATAR_FALLBACK_CODE_POINT = exports.USER_AVATAR_FALLBACK_CODE_POINT = 97; // code point for lowercase "a"
const USER_AVATAR_MAX_INITIALS = exports.USER_AVATAR_MAX_INITIALS = 2;

/**
 * Validates if the provided color string is a valid hex color.
 * @param color - The color to validate.
 * @returns True if the color is a valid hex color, false otherwise.
 */
function isValidUserProfileAvatarColor(color) {
  if (!color || typeof color !== 'string') {
    return false;
  }
  return color != null && color !== '' && /^#/.test(color) && _chromaJs.default.valid(color);
}

/**
 * Determines the color for the provided user profile.
 * If a color is present on the user profile itself, then that is used.
 * Otherwise, a color is provided from EUI's Visualization Colors based on the display name.
 *
 * @param {UserProfileUserInfo} user User info
 * @param {UserProfileAvatarData} avatar User avatar
 */
function getUserAvatarColor(user, avatar) {
  const firstCodePoint = getUserDisplayName(user).codePointAt(0) || USER_AVATAR_FALLBACK_CODE_POINT;
  const avatarColor = avatar === null || avatar === void 0 ? void 0 : avatar.color;
  const isValidColor = isValidUserProfileAvatarColor(avatarColor);
  if (isValidColor) {
    return avatarColor;
  }
  return _eui.VISUALIZATION_COLORS[firstCodePoint % _eui.VISUALIZATION_COLORS.length];
}

/**
 * Determines the initials for the provided user profile.
 * If initials are present on the user profile itself, then that is used.
 * Otherwise, the initials are calculated based off the words in the display name, with a max length of 2 characters.
 *
 * @param {UserProfileUserInfo} user User info
 * @param {UserProfileAvatarData} avatar User avatar
 */
function getUserAvatarInitials(user, avatar) {
  if (avatar && avatar.initials) {
    return avatar.initials;
  }
  const words = getUserDisplayName(user).split(' ');
  const numInitials = Math.min(USER_AVATAR_MAX_INITIALS, words.length);
  words.splice(numInitials, words.length);
  return words.map(word => word.substring(0, 1)).join('');
}

/**
 * Set of available name-related fields to pick as display name.
 */

/**
 * Determines the display name for the provided user information.
 * @param params Set of available user's name-related fields.
 */
function getUserDisplayName(params) {
  return params.full_name || params.email || params.username;
}

/**
 * Determines the display label for the provided user information.
 * Includes the email if it is different from the display name.
 * @param params Set of available user's name-related fields.
 */
function getUserDisplayLabel(user) {
  const displayName = getUserDisplayName(user);
  if (user.email && user.email !== displayName) {
    return `${displayName} (${user.email})`;
  }
  return displayName;
}