"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UptimeEsClient = void 0;
exports.createEsParams = createEsParams;
exports.shouldAppendSyntheticsIndex = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/core/server");
var _common = require("@kbn/inspector-plugin/common");
var _common2 = require("@kbn/observability-plugin/common");
var _common3 = require("@kbn/observability-shared-plugin/common");
var _preload = _interopRequireDefault(require("semver/preload"));
var _settings2 = require("../../constants/settings");
var _uptime_settings = require("./saved_objects/uptime_settings");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class UptimeEsClient {
  constructor(savedObjectsClient, esClient, options) {
    (0, _defineProperty2.default)(this, "isDev", void 0);
    (0, _defineProperty2.default)(this, "request", void 0);
    (0, _defineProperty2.default)(this, "baseESClient", void 0);
    (0, _defineProperty2.default)(this, "heartbeatIndices", void 0);
    (0, _defineProperty2.default)(this, "isInspectorEnabled", void 0);
    (0, _defineProperty2.default)(this, "inspectableEsQueries", []);
    (0, _defineProperty2.default)(this, "uiSettings", void 0);
    (0, _defineProperty2.default)(this, "savedObjectsClient", void 0);
    (0, _defineProperty2.default)(this, "isLegacyAlert", void 0);
    (0, _defineProperty2.default)(this, "stackVersion", void 0);
    const {
      stackVersion,
      isDev = false,
      uiSettings,
      request,
      heartbeatIndices = ''
    } = options !== null && options !== void 0 ? options : {};
    this.stackVersion = stackVersion;
    this.uiSettings = uiSettings;
    this.baseESClient = esClient;
    this.savedObjectsClient = savedObjectsClient;
    this.request = request;
    this.heartbeatIndices = heartbeatIndices;
    this.isDev = isDev;
    this.inspectableEsQueries = [];
    this.getInspectEnabled().catch(() => {});
  }
  async initSettings() {
    const self = this;
    const heartbeatIndices = await this.getIndices();
    self.heartbeatIndices = heartbeatIndices || '';
  }
  async search(params, operationName, index) {
    let res;
    let esError;
    await this.initSettings();
    const esParams = {
      index: index !== null && index !== void 0 ? index : this.heartbeatIndices,
      ...params
    };
    const startTimeNow = Date.now();
    let esRequestStatus = _common.RequestStatus.PENDING;
    try {
      res = await this.baseESClient.search(esParams, {
        meta: true,
        context: {
          loggingOptions: getElasticsearchRequestLoggingOptions()
        }
      });
      esRequestStatus = _common.RequestStatus.OK;
    } catch (e) {
      esError = e;
      esRequestStatus = _common.RequestStatus.ERROR;
    }
    if (this.request) {
      var _res;
      this.inspectableEsQueries.push((0, _common3.getInspectResponse)({
        esError,
        esRequestParams: esParams,
        esRequestStatus,
        esResponse: (_res = res) === null || _res === void 0 ? void 0 : _res.body,
        kibanaRequest: this.request,
        operationName: operationName !== null && operationName !== void 0 ? operationName : '',
        startTime: startTimeNow
      }));
    }
    if (esError) {
      throw esError;
    }
    return res;
  }
  async count(params) {
    let res;
    let esError;
    await this.initSettings();
    const esParams = {
      index: this.heartbeatIndices,
      ...params
    };
    try {
      res = await this.baseESClient.count(esParams, {
        meta: true,
        context: {
          loggingOptions: getElasticsearchRequestLoggingOptions()
        }
      });
    } catch (e) {
      esError = e;
    }
    if (esError) {
      throw esError;
    }
    return {
      result: res,
      indices: this.heartbeatIndices
    };
  }
  getSavedObjectsClient() {
    return this.savedObjectsClient;
  }
  async getInspectData(path) {
    const isInspectorEnabled = await this.getInspectEnabled();
    const showInspectData = (isInspectorEnabled || this.isDev) && path !== _constants.API_URLS.DYNAMIC_SETTINGS;
    if (showInspectData && this.inspectableEsQueries.length > 0) {
      return {
        _inspect: this.inspectableEsQueries
      };
    }
    return {};
  }
  async getInspectEnabled() {
    if (this.isInspectorEnabled !== undefined) {
      return this.isInspectorEnabled;
    }
    if (!this.uiSettings) {
      return false;
    }
    this.isInspectorEnabled = this.uiSettings.client.get(_common2.enableInspectEsQueries);
  }
  async getIndices() {
    // if isLegacyAlert appends synthetics-* if it's not already there
    let indices = '';
    let syntheticsIndexRemoved = false;
    let settingsChangedByUser = true;
    let settings = _settings2.DYNAMIC_SETTINGS_DEFAULT_ATTRIBUTES;
    if (this.heartbeatIndices) {
      indices = this.heartbeatIndices;
    } else {
      var _settings, _settings$syntheticsI;
      try {
        const obj = await this.savedObjectsClient.get(_uptime_settings.umDynamicSettings.name, _uptime_settings.settingsObjectId);
        settings = obj.attributes;
      } catch (getErr) {
        if (_server.SavedObjectsErrorHelpers.isNotFoundError(getErr)) {
          settingsChangedByUser = false;
        }
      }
      indices = ((_settings = settings) === null || _settings === void 0 ? void 0 : _settings.heartbeatIndices) || '';
      syntheticsIndexRemoved = (_settings$syntheticsI = settings.syntheticsIndexRemoved) !== null && _settings$syntheticsI !== void 0 ? _settings$syntheticsI : false;
    }
    if (indices.includes('synthetics-')) {
      return indices;
    }
    const appendSyntheticsIndex = shouldAppendSyntheticsIndex(this.stackVersion);
    if (appendSyntheticsIndex && (syntheticsIndexRemoved || !settingsChangedByUser)) {
      indices = indices + ',synthetics-*';
    }
    return indices;
  }
}
exports.UptimeEsClient = UptimeEsClient;
const shouldAppendSyntheticsIndex = stackVersion => {
  if (!stackVersion) {
    return false;
  }
  return _preload.default.lt(stackVersion, '8.10.0');
};
exports.shouldAppendSyntheticsIndex = shouldAppendSyntheticsIndex;
function createEsParams(params) {
  return params;
}
function getElasticsearchRequestLoggingOptions() {
  return {
    loggerName: 'uptime'
  };
}