"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerMigrateDataStreamRoutes = registerMigrateDataStreamRoutes;
var _configSchema = require("@kbn/config-schema");
var _elasticsearch = require("@elastic/elasticsearch");
var _i18n = require("@kbn/i18n");
var _upgradeAssistantPkgServer = require("@kbn/upgrade-assistant-pkg-server");
var _error = require("../../lib/data_streams/error");
var _constants = require("../../../common/constants");
var _data_streams = require("../../lib/data_streams");
var _map_any_error_to_kibana_http_response = require("./map_any_error_to_kibana_http_response");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerMigrateDataStreamRoutes({
  router,
  licensing,
  log,
  lib: {
    handleEsError
  },
  current
}) {
  const BASE_PATH = `${_constants.API_BASE_PATH}/migrate_data_stream`;
  router.get({
    path: `${BASE_PATH}/{dataStreamName}`,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on elasticsearch for authorization'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        dataStreamName: _configSchema.schema.string()
      })
    }
  }, (0, _upgradeAssistantPkgServer.versionCheckHandlerWrapper)(current.major)(async ({
    core
  }, request, response) => {
    const {
      elasticsearch: {
        client: esClient
      }
    } = await core;
    const {
      dataStreamName
    } = request.params;
    const asCurrentUser = esClient.asCurrentUser;
    const migrationService = (0, _data_streams.dataStreamMigrationServiceFactory)({
      esClient: asCurrentUser,
      log,
      licensing
    });
    try {
      const hasRequiredPrivileges = await migrationService.hasRequiredPrivileges(dataStreamName);
      const warnings = await migrationService.detectMigrationWarnings(dataStreamName);
      const migrationOp = await migrationService.fetchMigrationStatus(dataStreamName);
      const body = {
        migrationOp,
        warnings,
        hasRequiredPrivileges
      };
      return response.ok({
        body
      });
    } catch (err) {
      if (err instanceof _elasticsearch.errors.ResponseError) {
        return handleEsError({
          error: err,
          response
        });
      }
      return (0, _map_any_error_to_kibana_http_response.mapAnyErrorToKibanaHttpResponse)(_error.error);
    }
  }));
  router.post({
    path: `${BASE_PATH}/{dataStreamName}/reindex`,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on elasticsearch for authorization'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        dataStreamName: _configSchema.schema.string()
      })
    }
  }, (0, _upgradeAssistantPkgServer.versionCheckHandlerWrapper)(current.major)(async ({
    core
  }, request, response) => {
    const {
      elasticsearch: {
        client: esClient
      }
    } = await core;
    const {
      dataStreamName
    } = request.params;
    try {
      const callAsCurrentUser = esClient.asCurrentUser;
      const migrationService = (0, _data_streams.dataStreamMigrationServiceFactory)({
        esClient: callAsCurrentUser,
        log,
        licensing
      });
      if (!(await migrationService.hasRequiredPrivileges(dataStreamName))) {
        throw _error.error.accessForbidden(_i18n.i18n.translate('xpack.upgradeAssistant.datastream.reindexPrivilegesErrorBatch', {
          defaultMessage: `You do not have adequate privileges to reindex "{dataStreamName}".`,
          values: {
            dataStreamName
          }
        }));
      }
      await migrationService.createReindexOperation(dataStreamName);
      return response.ok();
    } catch (err) {
      if (err instanceof _elasticsearch.errors.ResponseError) {
        return handleEsError({
          error: err,
          response
        });
      }
      return (0, _map_any_error_to_kibana_http_response.mapAnyErrorToKibanaHttpResponse)(err);
    }
  }));
  router.get({
    path: `${BASE_PATH}/{dataStreamName}/metadata`,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on elasticsearch for authorization'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        dataStreamName: _configSchema.schema.string()
      })
    }
  }, (0, _upgradeAssistantPkgServer.versionCheckHandlerWrapper)(current.major)(async ({
    core
  }, request, response) => {
    const {
      elasticsearch: {
        client: esClient
      }
    } = await core;
    const {
      dataStreamName
    } = request.params;
    const asCurrentUser = esClient.asCurrentUser;
    const migrationService = (0, _data_streams.dataStreamMigrationServiceFactory)({
      esClient: asCurrentUser,
      log,
      licensing
    });
    try {
      const dataStreamMetadata = await migrationService.getDataStreamMetadata(dataStreamName);
      return response.ok({
        body: dataStreamMetadata || undefined
      });
    } catch (err) {
      if (err instanceof _elasticsearch.errors.ResponseError) {
        return handleEsError({
          error: err,
          response
        });
      }
      return (0, _map_any_error_to_kibana_http_response.mapAnyErrorToKibanaHttpResponse)(_error.error);
    }
  }));
  router.post({
    path: `${BASE_PATH}/{dataStreamName}/reindex/cancel`,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on elasticsearch for authorization'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        dataStreamName: _configSchema.schema.string()
      })
    }
  }, (0, _upgradeAssistantPkgServer.versionCheckHandlerWrapper)(current.major)(async ({
    core
  }, request, response) => {
    const {
      elasticsearch: {
        client: esClient
      }
    } = await core;
    const {
      dataStreamName
    } = request.params;
    const callAsCurrentUser = esClient.asCurrentUser;
    const migrationService = (0, _data_streams.dataStreamMigrationServiceFactory)({
      esClient: callAsCurrentUser,
      log,
      licensing
    });
    try {
      if (!(await migrationService.hasRequiredPrivileges(dataStreamName))) {
        throw _error.error.accessForbidden(_i18n.i18n.translate('xpack.upgradeAssistant.datastream.reindexPrivilegesErrorBatch', {
          defaultMessage: `You do not have adequate privileges to cancel reindexing "{dataStreamName}".`,
          values: {
            dataStreamName
          }
        }));
      }
      await migrationService.cancelReindexing(dataStreamName);
      return response.ok({
        body: {
          acknowledged: true
        }
      });
    } catch (err) {
      if (err instanceof _elasticsearch.errors.ResponseError) {
        return handleEsError({
          error: err,
          response
        });
      }
      return (0, _map_any_error_to_kibana_http_response.mapAnyErrorToKibanaHttpResponse)(_error.error);
    }
  }));
  router.post({
    path: `${BASE_PATH}/{dataStreamName}/readonly`,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on elasticsearch for authorization'
      }
    },
    options: {
      access: 'public',
      summary: `Set data stream indices as read-only`
    },
    validate: {
      body: _configSchema.schema.object({
        indices: _configSchema.schema.arrayOf(_configSchema.schema.string())
      }),
      params: _configSchema.schema.object({
        dataStreamName: _configSchema.schema.string()
      })
    }
  }, (0, _upgradeAssistantPkgServer.versionCheckHandlerWrapper)(current.major)(async ({
    core
  }, request, response) => {
    const {
      elasticsearch: {
        client: esClient
      }
    } = await core;
    const {
      dataStreamName
    } = request.params;
    const {
      indices
    } = request.body;
    const callAsCurrentUser = esClient.asCurrentUser;
    const migrationService = (0, _data_streams.dataStreamMigrationServiceFactory)({
      esClient: callAsCurrentUser,
      log,
      licensing
    });
    try {
      if (!(await migrationService.hasRequiredPrivileges(dataStreamName))) {
        throw _error.error.accessForbidden(_i18n.i18n.translate('xpack.upgradeAssistant.datastream.readonlyPrivilegesErrorBatch', {
          defaultMessage: `You do not have adequate privileges to mark indices inside data stream as readonly "{dataStreamName}".`,
          values: {
            dataStreamName
          }
        }));
      }
      await migrationService.readonlyIndices(dataStreamName, indices);
      return response.ok({
        body: {
          acknowledged: true
        }
      });
    } catch (err) {
      if (err instanceof _elasticsearch.errors.ResponseError) {
        return handleEsError({
          error: err,
          response
        });
      }
      return (0, _map_any_error_to_kibana_http_response.mapAnyErrorToKibanaHttpResponse)(err);
    }
  }));
}