"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteTransforms = deleteTransforms;
var _delete = require("@kbn/ml-data-view-utils/actions/delete");
var _transform = require("../../../../common/types/transform");
var _constants = require("../../../../common/constants");
var _error_utils = require("../../utils/error_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function deleteTransforms(reqBody, ctx, response, dataViewsService) {
  const {
    transformsInfo
  } = reqBody;

  // Cast possible undefineds as booleans
  const deleteDestIndex = !!reqBody.deleteDestIndex;
  const deleteDestDataView = !!reqBody.deleteDestDataView;
  const shouldForceDelete = !!reqBody.forceDelete;
  const results = {};
  const coreContext = await ctx.core;
  const esClient = coreContext.elasticsearch.client;
  for (const transformInfo of transformsInfo) {
    let destinationIndex;
    const transformDeleted = {
      success: false
    };
    const destIndexDeleted = {
      success: false
    };
    let destDataViewDeleted = {
      success: false
    };
    const transformId = transformInfo.id;
    // force delete only if the transform has failed
    let needToForceDelete = false;
    try {
      if (transformInfo.state === _constants.TRANSFORM_STATE.FAILED) {
        needToForceDelete = true;
      }
      if (!shouldForceDelete) {
        // Grab destination index info to delete
        try {
          const body = await esClient.asCurrentUser.transform.getTransform({
            transform_id: transformId
          });
          const transformConfig = body.transforms[0];
          destinationIndex = transformConfig.dest.index;
        } catch (getTransformConfigError) {
          transformDeleted.error = getTransformConfigError.meta.body.error;
          results[transformId] = {
            transformDeleted,
            destIndexDeleted,
            destDataViewDeleted,
            destinationIndex
          };
          // No need to perform further delete attempts
          continue;
        }
      }

      // Delete the data view if there's a data view that matches the name of dest index
      if (destinationIndex && deleteDestDataView) {
        destDataViewDeleted = await (0, _delete.deleteDataViewFn)({
          dataViewsService,
          dataViewName: destinationIndex
        });
      }
      try {
        await esClient.asCurrentUser.transform.deleteTransform({
          transform_id: transformId,
          force: shouldForceDelete && needToForceDelete,
          delete_dest_index: deleteDestIndex
        });
        transformDeleted.success = true;
        destIndexDeleted.success = deleteDestIndex;
      } catch (deleteTransformJobError) {
        transformDeleted.error = deleteTransformJobError.meta.body.error;
        if (deleteTransformJobError.statusCode === 403) {
          return response.forbidden();
        }
      }
      results[transformId] = {
        transformDeleted,
        destIndexDeleted,
        destDataViewDeleted,
        destinationIndex
      };
    } catch (e) {
      if ((0, _error_utils.isRequestTimeout)(e)) {
        return (0, _error_utils.fillResultsWithTimeouts)({
          results,
          id: transformInfo.id,
          items: transformsInfo,
          action: _transform.TRANSFORM_ACTIONS.DELETE
        });
      }
      results[transformId] = {
        transformDeleted: {
          success: false,
          error: e.meta.body.error
        }
      };
    }
  }
  return results;
}