"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformHealthServiceProvider = transformHealthServiceProvider;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _common = require("@kbn/field-formats-plugin/common");
var _constants = require("../../../../common/constants");
var _alerts = require("../../../../common/utils/alerts");
var _transform = require("../../../../common/types/transform");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const maxPathComponentLength = 2000;
const TRANSFORM_PAGE_SIZE = 1000;

/** Number of transforms IDs mentioned in the context message */
const TRANSFORMS_IDS_MESSAGE_LIMIT = 10;
function transformHealthServiceProvider({
  esClient,
  rulesClient,
  fieldFormatsRegistry
}) {
  const transformsDict = new Map();

  /**
   * Resolves result transform selection. Only continuously running transforms are included.
   * @param includeTransforms
   * @param excludeTransforms
   */
  const getResultsTransformIds = (transforms, includeTransforms, excludeTransforms) => {
    const continuousTransforms = transforms.filter(_transform.isContinuousTransform);
    continuousTransforms.forEach(t => {
      transformsDict.set(t.id, t);
    });
    return new Set(continuousTransforms.filter(t => includeTransforms.some(includedTransformId => new RegExp('^' + includedTransformId.replace(/\*/g, '.*') + '$').test(t.id)) && (Array.isArray(excludeTransforms) && excludeTransforms.length > 0 ? excludeTransforms.every(excludedTransformId => new RegExp('^' + excludedTransformId.replace(/\*/g, '.*') + '$').test(t.id) === false) : true)).map(t => t.id));
  };

  /**
   * Returns a string with transform IDs for the context message.
   */
  const getContextMessageTransformIds = transformIds => {
    const count = transformIds.length;
    let transformsString = transformIds.join(', ');
    if (transformIds.length > TRANSFORMS_IDS_MESSAGE_LIMIT) {
      transformsString = _i18n.i18n.translate('xpack.transform.alertTypes.transformHealth.truncatedTransformIdsMessage', {
        defaultMessage: '{truncatedTransformIds} and {restCount, plural, one {# other} other {# others}}',
        values: {
          truncatedTransformIds: transformIds.slice(0, TRANSFORMS_IDS_MESSAGE_LIMIT).join(', '),
          restCount: count - TRANSFORMS_IDS_MESSAGE_LIMIT
        }
      });
    }
    return transformsString;
  };
  const getTransformStats = (0, _lodash.memoize)(async transformIds => {
    const transformIdsString = Array.from(transformIds).join(',');
    if (transformIdsString.length < maxPathComponentLength) {
      return (await esClient.transform.getTransformStats({
        transform_id: transformIdsString,
        // @ts-expect-error `basic` query option not yet in @elastic/elasticsearch
        basic: true,
        size: transformIds.size
      })).transforms;
    } else {
      // Fetch all transforms and filter out the ones that are not in the list.
      return (await esClient.transform.getTransformStats({
        // @ts-expect-error `basic` query option not yet in @elastic/elasticsearch
        basic: true,
        transform_id: '_all',
        size: TRANSFORM_PAGE_SIZE
      })).transforms.filter(t => transformIds.has(t.id));
    }
  });
  function baseTransformAlertResponseFormatter(transformStats) {
    var _transformsDict$get, _transformStats$node, _transformStats$healt, _transformStats$healt2;
    const dateFormatter = fieldFormatsRegistry.deserialize({
      id: _common.FIELD_FORMAT_IDS.DATE
    });
    return {
      transform_id: transformStats.id,
      description: (_transformsDict$get = transformsDict.get(transformStats.id)) === null || _transformsDict$get === void 0 ? void 0 : _transformsDict$get.description,
      transform_state: transformStats.state,
      node_name: (_transformStats$node = transformStats.node) === null || _transformStats$node === void 0 ? void 0 : _transformStats$node.name,
      health_status: (0, _constants.mapEsHealthStatus2TransformHealthStatus)((_transformStats$healt = transformStats.health) === null || _transformStats$healt === void 0 ? void 0 : _transformStats$healt.status),
      ...((_transformStats$healt2 = transformStats.health) !== null && _transformStats$healt2 !== void 0 && _transformStats$healt2.issues ? {
        issues: transformStats.health.issues.map(issue => {
          return {
            issue: issue.issue,
            details: issue.details,
            count: issue.count,
            ...(issue.first_occurrence ? {
              first_occurrence: dateFormatter.convert(issue.first_occurrence)
            } : {})
          };
        })
      } : {})
    };
  }
  return {
    /**
     * Returns report about not started transforms
     * @param transformIds
     *
     * @return - Partitions with not started and started transforms
     */
    async getTransformsStateReport(transformIds) {
      const transformsStats = await getTransformStats(transformIds);
      return (0, _lodash.partition)(transformsStats.map(baseTransformAlertResponseFormatter), t => t.transform_state !== _constants.TRANSFORM_STATE.STARTED && t.transform_state !== _constants.TRANSFORM_STATE.INDEXING);
    },
    /**
     * Returns report about transforms that contain error messages
     * @deprecated This health check is no longer in use
     * @param transformIds
     */
    async getErrorMessagesReport(transformIds) {
      var _response$aggregation;
      const response = await esClient.search({
        index: _constants.TRANSFORM_NOTIFICATIONS_INDEX,
        size: 0,
        query: {
          bool: {
            filter: [{
              term: {
                level: 'error'
              }
            }, {
              terms: {
                transform_id: Array.from(transformIds)
              }
            }]
          }
        },
        aggs: {
          by_transform: {
            terms: {
              field: 'transform_id',
              size: transformIds.size
            },
            aggs: {
              error_messages: {
                top_hits: {
                  size: 10,
                  _source: {
                    includes: ['message', 'level', 'timestamp', 'node_name']
                  }
                }
              }
            }
          }
        }
      });

      // If transform contains errors, it's in a failed state
      const transformsStats = await getTransformStats(transformIds);
      const failedTransforms = new Set(transformsStats.filter(t => t.state === _constants.TRANSFORM_STATE.FAILED).map(t => t.id));
      return ((_response$aggregation = response.aggregations) === null || _response$aggregation === void 0 ? void 0 : _response$aggregation.by_transform.buckets).map(({
        key,
        error_messages: errorMessages
      }) => {
        return {
          transform_id: key,
          error_messages: errorMessages.hits.hits.map(v => v._source)
        };
      }).filter(v => failedTransforms.has(v.transform_id));
    },
    /**
     * Returns report about unhealthy transforms
     * @param transformIds
     */
    async getUnhealthyTransformsReport(transformIds) {
      const transformsStats = await getTransformStats(transformIds);
      return transformsStats.filter(t => {
        var _t$health;
        return (0, _constants.mapEsHealthStatus2TransformHealthStatus)((_t$health = t.health) === null || _t$health === void 0 ? void 0 : _t$health.status) !== _constants.TRANSFORM_HEALTH_STATUS.green;
      }).map(baseTransformAlertResponseFormatter);
    },
    /**
     * Returns results of the transform health checks
     * @param params
     */
    async getHealthChecksResults(params, previousState) {
      const includeAll = params.includeTransforms.some(id => id === _constants.ALL_TRANSFORMS_SELECTION);
      const transforms = (await esClient.transform.getTransform({
        ...(includeAll ? {} : {
          transform_id: params.includeTransforms.join(',')
        }),
        allow_no_match: true,
        size: TRANSFORM_PAGE_SIZE
      })).transforms;
      const transformIds = getResultsTransformIds(transforms, params.includeTransforms, params.excludeTransforms);
      const testsConfig = (0, _alerts.getResultTestConfig)(params.testsConfig);
      const result = [];
      if (testsConfig.notStarted.enabled) {
        var _previousState$notSta;
        const [notStartedTransform, startedTransforms] = await this.getTransformsStateReport(transformIds);
        const prevNotStartedSet = new Set((_previousState$notSta = previousState === null || previousState === void 0 ? void 0 : previousState.notStarted) !== null && _previousState$notSta !== void 0 ? _previousState$notSta : []);
        const recoveredTransforms = startedTransforms.filter(t => prevNotStartedSet.has(t.transform_id));
        const isHealthy = notStartedTransform.length === 0;

        // if healthy, mention transforms that were not started
        const count = isHealthy ? recoveredTransforms.length : notStartedTransform.length;
        const transformsString = getContextMessageTransformIds((isHealthy ? recoveredTransforms : notStartedTransform).map(t => t.transform_id));
        result.push({
          isHealthy,
          name: _constants.TRANSFORM_HEALTH_CHECK_NAMES.notStarted.name,
          context: {
            results: isHealthy ? recoveredTransforms : notStartedTransform,
            message: isHealthy ? _i18n.i18n.translate('xpack.transform.alertTypes.transformHealth.notStartedRecoveryMessage', {
              defaultMessage: '{count, plural, =0 {All transforms are started} one {Transform {transformsString} is started} other {# transforms are started: {transformsString}}}.',
              values: {
                count,
                transformsString
              }
            }) : _i18n.i18n.translate('xpack.transform.alertTypes.transformHealth.notStartedMessage', {
              defaultMessage: '{count, plural, one {Transform {transformsString} is not started} other {# transforms are not started: {transformsString}}}.',
              values: {
                count,
                transformsString
              }
            })
          }
        });
      }
      if (testsConfig.errorMessages.enabled) {
        const response = await this.getErrorMessagesReport(transformIds);
        const isHealthy = response.length === 0;
        const count = response.length;
        const transformsString = response.map(t => t.transform_id).join(', ');
        result.push({
          isHealthy,
          name: _constants.TRANSFORM_HEALTH_CHECK_NAMES.errorMessages.name,
          context: {
            results: isHealthy ? [] : response,
            message: isHealthy ? _i18n.i18n.translate('xpack.transform.alertTypes.transformHealth.errorMessagesRecoveryMessage', {
              defaultMessage: 'No errors in the {count, plural, one {transform} other {transforms}} messages.',
              values: {
                count: transformIds.size
              }
            }) : _i18n.i18n.translate('xpack.transform.alertTypes.transformHealth.errorMessagesMessage', {
              defaultMessage: '{count, plural, one {Transform} other {Transforms}} {transformsString} {count, plural, one {contains} other {contain}} error messages.',
              values: {
                count,
                transformsString
              }
            })
          }
        });
      }
      if (testsConfig.healthCheck.enabled) {
        var _previousState$unheal, _previousState$unheal2, _previousState$unheal3;
        const response = await this.getUnhealthyTransformsReport(transformIds);
        const isHealthy = response.length === 0;
        const count = isHealthy ? (_previousState$unheal = previousState === null || previousState === void 0 ? void 0 : (_previousState$unheal2 = previousState.unhealthy) === null || _previousState$unheal2 === void 0 ? void 0 : _previousState$unheal2.length) !== null && _previousState$unheal !== void 0 ? _previousState$unheal : 0 : response.length;
        const transformsString = getContextMessageTransformIds(isHealthy ? (_previousState$unheal3 = previousState === null || previousState === void 0 ? void 0 : previousState.unhealthy) !== null && _previousState$unheal3 !== void 0 ? _previousState$unheal3 : [] : response.map(t => t.transform_id));
        result.push({
          isHealthy,
          name: _constants.TRANSFORM_HEALTH_CHECK_NAMES.healthCheck.name,
          context: {
            results: isHealthy ? [] : response,
            message: isHealthy ? _i18n.i18n.translate('xpack.transform.alertTypes.transformHealth.healthCheckRecoveryMessage', {
              defaultMessage: '{count, plural, =0 {All transforms are healthy} one {Transform {transformsString} is healthy} other {# transforms are healthy: {transformsString}}}.',
              values: {
                count,
                transformsString
              }
            }) : _i18n.i18n.translate('xpack.transform.alertTypes.transformHealth.healthCheckMessage', {
              defaultMessage: '{count, plural, one {Transform {transformsString} is unhealthy} other {# transforms are unhealthy: {transformsString}}}.',
              values: {
                count,
                transformsString
              }
            })
          }
        });
      }
      return result;
    },
    /**
     * Updates transform list with associated alerting rules.
     */
    async populateTransformsWithAssignedRules(transforms) {
      const continuousTransforms = transforms.filter(_transform.isContinuousTransform);
      if (!rulesClient) {
        throw new Error('Rules client is missing');
      }
      if (!continuousTransforms.length) {
        return transforms;
      }
      const transformMap = (0, _lodash.keyBy)(continuousTransforms, 'id');
      const transformAlertingRules = await rulesClient.find({
        options: {
          perPage: 1000,
          filter: `alert.attributes.alertTypeId:${_constants.TRANSFORM_RULE_TYPE.TRANSFORM_HEALTH}`
        }
      });
      for (const ruleInstance of transformAlertingRules.data) {
        // Retrieve result transform IDs
        const {
          includeTransforms,
          excludeTransforms
        } = ruleInstance.params;
        const resultTransformIds = getResultsTransformIds(transforms, includeTransforms, excludeTransforms);
        resultTransformIds.forEach(transformId => {
          const transformRef = transformMap[transformId];
          if (transformRef) {
            if (Array.isArray(transformRef.alerting_rules)) {
              transformRef.alerting_rules.push(ruleInstance);
            } else {
              transformRef.alerting_rules = [ruleInstance];
            }
          }
        });
      }
      return continuousTransforms;
    }
  };
}