"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.claimAvailableTasksUpdateByQuery = claimAvailableTasksUpdateByQuery;
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _lodash = require("lodash");
var _result_type = require("../lib/result_type");
var _ = require(".");
var _task_running = require("../task_running");
var _task_claiming = require("../queries/task_claiming");
var _task_events = require("../task_events");
var _query_clauses = require("../queries/query_clauses");
var _mark_available_tasks_as_claimed = require("../queries/mark_available_tasks_as_claimed");
var _task_store = require("../task_store");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * This module contains helpers for managing the task manager storage layer.
 */

async function claimAvailableTasksUpdateByQuery(opts) {
  const {
    getCapacity,
    claimOwnershipUntil,
    batches,
    events$,
    taskStore
  } = opts;
  const {
    definitions,
    excludedTaskTypes,
    taskMaxAttempts
  } = opts;
  const initialCapacity = getCapacity();
  let accumulatedResult = (0, _.getEmptyClaimOwnershipResult)();
  const stopTaskTimer = (0, _task_events.startTaskTimer)();
  for (const batch of batches) {
    const capacity = Math.min(initialCapacity - accumulatedResult.stats.tasksClaimed, (0, _task_claiming.isLimited)(batch) ? getCapacity(batch.tasksTypes) : getCapacity());

    // if we have no more capacity, short circuit here
    if (capacity <= 0) {
      return accumulatedResult;
    }
    const result = await executeClaimAvailableTasks({
      claimOwnershipUntil,
      size: capacity,
      events$,
      taskTypes: (0, _task_claiming.isLimited)(batch) ? new Set([batch.tasksTypes]) : batch.tasksTypes,
      taskStore,
      definitions,
      excludedTaskTypes,
      taskMaxAttempts
    });
    accumulatedResult = accumulateClaimOwnershipResults(accumulatedResult, result);
    accumulatedResult.stats.tasksConflicted = (0, _task_store.correctVersionConflictsForContinuation)(accumulatedResult.stats.tasksClaimed, accumulatedResult.stats.tasksConflicted, initialCapacity);
  }
  return {
    ...accumulatedResult,
    timing: stopTaskTimer()
  };
}
async function executeClaimAvailableTasks(opts) {
  const {
    taskStore,
    size,
    taskTypes,
    events$,
    definitions
  } = opts;
  const {
    updated: tasksUpdated,
    version_conflicts: tasksConflicted
  } = await markAvailableTasksAsClaimed(opts);
  const docs = tasksUpdated > 0 ? await sweepForClaimedTasks(taskStore, taskTypes, size, definitions) : [];
  emitEvents(events$, docs.map(doc => (0, _task_events.asTaskClaimEvent)(doc.id, (0, _result_type.asOk)(doc))));
  const stats = {
    tasksUpdated,
    tasksConflicted,
    tasksClaimed: docs.length
  };
  return {
    stats,
    docs
  };
}
function emitEvents(events$, events) {
  events.forEach(event => events$.next(event));
}
async function markAvailableTasksAsClaimed({
  definitions,
  excludedTaskTypes,
  taskStore,
  claimOwnershipUntil,
  size,
  taskTypes,
  taskMaxAttempts
}) {
  const {
    taskTypesToSkip = [],
    taskTypesToClaim = []
  } = (0, _lodash.groupBy)(definitions.getAllTypes(), type => taskTypes.has(type) && !(0, _.isTaskTypeExcluded)(excludedTaskTypes, type) ? 'taskTypesToClaim' : 'taskTypesToSkip');
  const queryForScheduledTasks = (0, _query_clauses.mustBeAllOf)(
  // Task must be enabled
  _mark_available_tasks_as_claimed.EnabledTask,
  // Either a task with idle status and runAt <= now or
  // status running or claiming with a retryAt <= now.
  (0, _query_clauses.shouldBeOneOf)(_mark_available_tasks_as_claimed.IdleTaskWithExpiredRunAt, _mark_available_tasks_as_claimed.RunningOrClaimingTaskWithExpiredRetryAt));
  const sort = (0, _mark_available_tasks_as_claimed.getClaimSort)(definitions);
  const query = (0, _query_clauses.matchesClauses)(queryForScheduledTasks, (0, _query_clauses.filterDownBy)(_mark_available_tasks_as_claimed.InactiveTasks));
  const script = (0, _mark_available_tasks_as_claimed.updateFieldsAndMarkAsFailed)({
    fieldUpdates: {
      ownerId: taskStore.taskManagerId,
      retryAt: claimOwnershipUntil
    },
    claimableTaskTypes: taskTypesToClaim,
    skippedTaskTypes: taskTypesToSkip,
    taskMaxAttempts: (0, _lodash.pick)(taskMaxAttempts, taskTypesToClaim)
  });
  const apmTrans = _elasticApmNode.default.startTransaction(_task_claiming.TASK_MANAGER_MARK_AS_CLAIMED, _task_running.TASK_MANAGER_TRANSACTION_TYPE);
  try {
    const result = await taskStore.updateByQuery({
      query,
      script,
      sort
    }, {
      max_docs: size
    });
    apmTrans.end('success');
    return result;
  } catch (err) {
    apmTrans.end('failure');
    throw err;
  }
}
async function sweepForClaimedTasks(taskStore, taskTypes, size, definitions) {
  const claimedTasksQuery = (0, _mark_available_tasks_as_claimed.tasksClaimedByOwner)(taskStore.taskManagerId, (0, _mark_available_tasks_as_claimed.tasksOfType)([...taskTypes]));
  const {
    docs
  } = await taskStore.fetch({
    query: claimedTasksQuery,
    size,
    sort: (0, _mark_available_tasks_as_claimed.getClaimSort)(definitions),
    seq_no_primary_term: true
  });
  return docs;
}
function accumulateClaimOwnershipResults(prev = (0, _.getEmptyClaimOwnershipResult)(), next) {
  if (next) {
    const {
      stats,
      docs,
      timing
    } = next;
    const res = {
      stats: {
        tasksUpdated: stats.tasksUpdated + prev.stats.tasksUpdated,
        tasksConflicted: stats.tasksConflicted + prev.stats.tasksConflicted,
        tasksClaimed: stats.tasksClaimed + prev.stats.tasksClaimed
      },
      docs: [...prev.docs, ...docs],
      timing
    };
    return res;
  }
  return prev;
}