"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BufferedTaskStore = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _bulk_operation_buffer = require("./lib/bulk_operation_buffer");
var _result_type = require("./lib/result_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// by default allow updates to be buffered for up to 50ms
const DEFAULT_BUFFER_MAX_DURATION = 50;
class BufferedTaskStore {
  constructor(taskStore, options) {
    (0, _defineProperty2.default)(this, "bufferedPartialUpdate", void 0);
    (0, _defineProperty2.default)(this, "bufferedUpdate", void 0);
    (0, _defineProperty2.default)(this, "bufferedRemove", void 0);
    this.taskStore = taskStore;
    this.bufferedUpdate = (0, _bulk_operation_buffer.createBuffer)(
    // Setting validate: false because we'll validate per update call
    //
    // Ideally we could accumulate the "validate" options and pass them
    // to .bulkUpdate per doc, but the required changes to the bulk_operation_buffer
    // to track the values are high and deffered for now.
    docs => taskStore.bulkUpdate(docs, {
      validate: false
    }), {
      bufferMaxDuration: DEFAULT_BUFFER_MAX_DURATION,
      ...options
    });
    this.bufferedPartialUpdate = (0, _bulk_operation_buffer.createBuffer)(docs => taskStore.bulkPartialUpdate(docs), {
      bufferMaxDuration: DEFAULT_BUFFER_MAX_DURATION,
      ...options
    });
    this.bufferedRemove = (0, _bulk_operation_buffer.createBuffer)(async ids => {
      const result = await taskStore.bulkRemove(ids.map(({
        id
      }) => id));
      return result.statuses.map(status => status.error ? (0, _result_type.asErr)({
        error: status.error,
        id: status.id,
        type: status.type
      }) : (0, _result_type.asOk)(status));
    }, {
      bufferMaxDuration: DEFAULT_BUFFER_MAX_DURATION,
      ...options
    });
  }
  async update(doc, options) {
    const docToUpdate = this.taskStore.taskValidator.getValidatedTaskInstanceForUpdating(doc, {
      validate: options.validate
    });
    const result = await (0, _result_type.unwrapPromise)(this.bufferedUpdate(docToUpdate));
    return this.taskStore.taskValidator.getValidatedTaskInstanceFromReading(result, {
      validate: options.validate
    });
  }
  async partialUpdate(partialDoc, options) {
    // merge the partial updates with the doc and validate
    const {
      stateVersion
    } = this.taskStore.taskValidator.getValidatedTaskInstanceForUpdating({
      ...options.doc,
      ...partialDoc
    }, {
      validate: options.validate
    });
    const result = await (0, _result_type.unwrapPromise)(this.bufferedPartialUpdate({
      ...partialDoc,
      ...(stateVersion ? {
        stateVersion
      } : {})
    }));

    // merge the partial update result with the doc and validate
    return this.taskStore.taskValidator.getValidatedTaskInstanceFromReading({
      ...options.doc,
      ...result
    }, {
      validate: options.validate
    });
  }
  async remove(id) {
    await (0, _result_type.unwrapPromise)(this.bufferedRemove({
      id
    }));
  }
}
exports.BufferedTaskStore = BufferedTaskStore;