"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runSynPrivateLocationMonitorsTaskSoon = exports.resetSyncPrivateCleanUpState = exports.disableSyncPrivateLocationTask = exports.SyncPrivateLocationMonitorsTask = exports.PRIVATE_LOCATIONS_SYNC_TASK_ID = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("@kbn/spaces-plugin/common/constants");
var _moment = _interopRequireDefault(require("moment"));
var _common = require("@kbn/alerting-plugin/common");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _deploy_private_location_monitors = require("./deploy_private_location_monitors");
var _clean_up_duplicate_policies = require("./clean_up_duplicate_policies");
var _get_private_locations = require("../synthetics_service/get_private_locations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TASK_TYPE = 'Synthetics:Sync-Private-Location-Monitors';
const PRIVATE_LOCATIONS_SYNC_TASK_ID = exports.PRIVATE_LOCATIONS_SYNC_TASK_ID = `${TASK_TYPE}-single-instance`;
const TASK_SCHEDULE = '60m';
class SyncPrivateLocationMonitorsTask {
  constructor(serverSetup, syntheticsMonitorClient) {
    (0, _defineProperty2.default)(this, "deployPackagePolicies", void 0);
    (0, _defineProperty2.default)(this, "start", async () => {
      const {
        pluginsStart: {
          taskManager
        }
      } = this.serverSetup;
      this.debugLog(`Scheduling private location task`);
      await taskManager.ensureScheduled({
        id: PRIVATE_LOCATIONS_SYNC_TASK_ID,
        state: {},
        schedule: {
          interval: TASK_SCHEDULE
        },
        taskType: TASK_TYPE,
        params: {}
      });
      this.debugLog(`Sync private location monitors task scheduled successfully`);
    });
    (0, _defineProperty2.default)(this, "debugLog", message => {
      this.serverSetup.logger.debug(`[SyncPrivateLocationMonitorsTask] ${message}`);
    });
    this.serverSetup = serverSetup;
    this.syntheticsMonitorClient = syntheticsMonitorClient;
    this.deployPackagePolicies = new _deploy_private_location_monitors.DeployPrivateLocationMonitors(serverSetup, syntheticsMonitorClient);
  }
  registerTaskDefinition(taskManager) {
    taskManager.registerTaskDefinitions({
      [TASK_TYPE]: {
        title: 'Synthetics Sync Global Params Task',
        description: 'This task is executed so that we can sync private location monitors for example when global params are updated',
        timeout: '10m',
        maxAttempts: 1,
        createTaskRunner: ({
          taskInstance
        }) => {
          return {
            run: async () => {
              return this.runTask({
                taskInstance
              });
            }
          };
        }
      }
    });
  }
  async runTask({
    taskInstance
  }) {
    this.debugLog(`Syncing private location monitors, current task state is ${JSON.stringify(taskInstance.state)}`);
    const {
      coreStart: {
        savedObjects
      },
      logger,
      pluginsStart: {
        encryptedSavedObjects
      }
    } = this.serverSetup;
    let lastStartedAt = taskInstance.state.lastStartedAt;
    // if it's too old, set it to 10 minutes ago to avoid syncing everything the first time
    if (!lastStartedAt || (0, _moment.default)(lastStartedAt).isBefore((0, _moment.default)().subtract(6, 'hour'))) {
      lastStartedAt = (0, _moment.default)().subtract(10, 'minute').toISOString();
    }
    const taskState = this.getNewTaskState({
      taskInstance
    });
    try {
      const soClient = savedObjects.createInternalRepository([_common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE]);
      const {
        performCleanupSync
      } = await this.cleanUpDuplicatedPackagePolicies(soClient, taskState);
      const allPrivateLocations = await (0, _get_private_locations.getPrivateLocations)(soClient, _constants.ALL_SPACES_ID);
      const {
        hasMWsChanged
      } = await this.hasMWsChanged({
        soClient,
        taskState,
        lastStartedAt
      });

      // Only perform syncGlobalParams if:
      // - hasMWsChanged and disableAutoSync is false
      // - OR performCleanupSync is true (from cleanup), regardless of disableAutoSync
      const dataChangeSync = hasMWsChanged && !taskState.disableAutoSync;
      if (dataChangeSync || performCleanupSync) {
        if (dataChangeSync) {
          this.debugLog(`Syncing private location monitors because data has changed`);
        } else if (performCleanupSync) {
          this.debugLog(`Syncing private location monitors because cleanup performed a change`);
        }
        if (allPrivateLocations.length > 0) {
          await this.deployPackagePolicies.syncPackagePolicies({
            allPrivateLocations,
            soClient,
            encryptedSavedObjects
          });
        } else {
          this.debugLog(`No private locations found, skipping sync`);
        }
        this.debugLog(`Sync of private location monitors succeeded`);
      } else {
        if (taskState.disableAutoSync) {
          this.debugLog(`Auto sync is disabled, skipping sync of private location monitors`);
        } else {
          this.debugLog(`No data has changed since last run ${lastStartedAt}, skipping sync of private location monitors`);
        }
      }
    } catch (error) {
      logger.error(`Sync of private location monitors failed: ${error.message}`);
      return {
        error,
        state: taskState,
        schedule: {
          interval: TASK_SCHEDULE
        }
      };
    }
    return {
      state: taskState,
      schedule: {
        interval: TASK_SCHEDULE
      }
    };
  }
  getNewTaskState({
    taskInstance
  }) {
    var _taskInstance$state$d;
    const startedAt = taskInstance.startedAt || new Date();
    return {
      lastStartedAt: startedAt.toISOString(),
      lastTotalMWs: taskInstance.state.lastTotalMWs || 0,
      hasAlreadyDoneCleanup: taskInstance.state.hasAlreadyDoneCleanup || false,
      maxCleanUpRetries: taskInstance.state.maxCleanUpRetries || 3,
      disableAutoSync: (_taskInstance$state$d = taskInstance.state.disableAutoSync) !== null && _taskInstance$state$d !== void 0 ? _taskInstance$state$d : false
    };
  }
  parseLocations(config) {
    const {
      locations
    } = config;
    const privateLocations = locations.filter(loc => !loc.isServiceManaged);
    const publicLocations = locations.filter(loc => loc.isServiceManaged);
    return {
      privateLocations,
      publicLocations
    };
  }
  async hasMWsChanged({
    soClient,
    lastStartedAt,
    taskState
  }) {
    const {
      logger
    } = this.serverSetup;
    const {
      lastTotalMWs
    } = taskState;
    const [editedMWs, totalMWs] = await Promise.all([soClient.find({
      type: _common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE,
      perPage: 0,
      namespaces: [_constants.ALL_SPACES_ID],
      filter: `${_common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE}.updated_at > "${lastStartedAt}"`,
      fields: []
    }), soClient.find({
      type: _common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE,
      perPage: 0,
      namespaces: [_constants.ALL_SPACES_ID],
      fields: []
    })]);
    logger.debug(`Found ${editedMWs.total} maintenance windows updated and ${totalMWs.total} total maintenance windows`);
    const updatedMWs = editedMWs.total;
    const noOfMWs = totalMWs.total;
    const hasMWsChanged = updatedMWs > 0 || noOfMWs !== lastTotalMWs;
    taskState.lastTotalMWs = noOfMWs;
    return {
      hasMWsChanged
    };
  }
  async cleanUpDuplicatedPackagePolicies(soClient, taskState) {
    return await (0, _clean_up_duplicate_policies.cleanUpDuplicatedPackagePolicies)(this.serverSetup, soClient, taskState);
  }
}
exports.SyncPrivateLocationMonitorsTask = SyncPrivateLocationMonitorsTask;
const runSynPrivateLocationMonitorsTaskSoon = async ({
  server,
  retries = 5
}) => {
  try {
    await (0, _pRetry.default)(async () => {
      const {
        logger,
        pluginsStart: {
          taskManager
        }
      } = server;
      logger.debug(`Scheduling Synthetics sync private location monitors task soon`);
      await taskManager.runSoon(PRIVATE_LOCATIONS_SYNC_TASK_ID);
      logger.debug(`Synthetics sync private location task scheduled successfully`);
    }, {
      retries
    });
  } catch (error) {
    server.logger.error(`Error scheduling Synthetics sync private location monitors task: ${error.message}`, {
      error
    });
  }
};
exports.runSynPrivateLocationMonitorsTaskSoon = runSynPrivateLocationMonitorsTaskSoon;
const resetSyncPrivateCleanUpState = async ({
  server
}) => {
  const {
    logger,
    pluginsStart: {
      taskManager
    }
  } = server;
  logger.debug(`Resetting Synthetics sync private location monitors cleanup state`);
  await taskManager.bulkUpdateState([PRIVATE_LOCATIONS_SYNC_TASK_ID], state => ({
    ...state,
    hasAlreadyDoneCleanup: false
  }));
  await runSynPrivateLocationMonitorsTaskSoon({
    server
  });
  logger.debug(`Synthetics sync private location monitors cleanup state reset successfully`);
};
exports.resetSyncPrivateCleanUpState = resetSyncPrivateCleanUpState;
const disableSyncPrivateLocationTask = async ({
  server,
  disableAutoSync
}) => {
  const {
    logger,
    pluginsStart: {
      taskManager
    }
  } = server;
  logger.debug(`Setting Synthetics sync private location monitors disableAutoSync to ${disableAutoSync}`);
  await taskManager.bulkUpdateState([PRIVATE_LOCATIONS_SYNC_TASK_ID], state => ({
    ...state,
    disableAutoSync
  }));
  logger.debug(`Synthetics sync private location monitors disableAutoSync set successfully`);
};
exports.disableSyncPrivateLocationTask = disableSyncPrivateLocationTask;