"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SyntheticsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _common = require("@kbn/spaces-plugin/common");
var _constants = require("@kbn/spaces-plugin/common/constants");
var _pMap = _interopRequireDefault(require("p-map"));
var _moment = _interopRequireDefault(require("moment"));
var _maintenance_window_client = require("@kbn/alerting-plugin/server/maintenance_window_client");
var _lodash = require("lodash");
var _common2 = require("@kbn/alerting-plugin/common");
var _clean_up_task = require("./private_location/clean_up_task");
var _saved_objects = require("../../common/types/saved_objects");
var _monitor_upgrade_sender = require("../routes/telemetry/monitor_upgrade_sender");
var _install_index_templates = require("../routes/synthetics_service/install_index_templates");
var _get_api_key = require("./get_api_key");
var _get_es_hosts = require("./get_es_hosts");
var _service_api_client = require("./service_api_client");
var _runtime_types = require("../../common/runtime_types");
var _get_service_locations = require("./get_service_locations");
var _secrets = require("./utils/secrets");
var _format_configs = require("./formatters/public_formatters/format_configs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable max-classes-per-file */

const SYNTHETICS_SERVICE_SYNC_MONITORS_TASK_TYPE = 'UPTIME:SyntheticsService:Sync-Saved-Monitor-Objects';
const SYNTHETICS_SERVICE_SYNC_MONITORS_TASK_ID = 'UPTIME:SyntheticsService:sync-task';
const SYNTHETICS_SERVICE_SYNC_INTERVAL_DEFAULT = '5m';
class SyntheticsService {
  constructor(server) {
    var _server$config$servic;
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "server", void 0);
    (0, _defineProperty2.default)(this, "apiClient", void 0);
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "esHosts", void 0);
    (0, _defineProperty2.default)(this, "locations", void 0);
    (0, _defineProperty2.default)(this, "throttling", void 0);
    (0, _defineProperty2.default)(this, "indexTemplateExists", void 0);
    (0, _defineProperty2.default)(this, "indexTemplateInstalling", void 0);
    (0, _defineProperty2.default)(this, "isAllowed", void 0);
    (0, _defineProperty2.default)(this, "signupUrl", void 0);
    (0, _defineProperty2.default)(this, "syncErrors", []);
    (0, _defineProperty2.default)(this, "invalidApiKeyError", void 0);
    this.logger = server.logger;
    this.server = server;
    this.config = (_server$config$servic = server.config.service) !== null && _server$config$servic !== void 0 ? _server$config$servic : {};

    // set isAllowed to false if manifestUrl is not set
    this.isAllowed = this.config.manifestUrl ? false : true;
    this.signupUrl = null;
    this.apiClient = new _service_api_client.ServiceAPIClient(server.logger, this.config, this.server);
    this.esHosts = (0, _get_es_hosts.getEsHosts)({
      config: this.config,
      cloud: server.cloud
    });
    this.locations = [];
  }
  async setup(taskManager) {
    this.registerSyncTask(taskManager);
    (0, _clean_up_task.registerCleanUpTask)(taskManager, this.server);
    await this.registerServiceLocations();
    const {
      allowed,
      signupUrl
    } = await this.apiClient.checkAccountAccessStatus();
    this.isAllowed = allowed;
    this.signupUrl = signupUrl;
  }
  start(taskManager) {
    var _this$config;
    if ((_this$config = this.config) !== null && _this$config !== void 0 && _this$config.manifestUrl) {
      void this.scheduleSyncTask(taskManager);
    }
    void this.setupIndexTemplates();
  }
  async setupIndexTemplates() {
    var _this$config2;
    if (process.env.CI && !((_this$config2 = this.config) !== null && _this$config2 !== void 0 && _this$config2.manifestUrl)) {
      // skip installation on CI
      return;
    }
    if (this.indexTemplateExists) {
      // if already installed, don't need to reinstall
      return;
    }
    try {
      if (!this.indexTemplateInstalling) {
        this.indexTemplateInstalling = true;
        const installedPackage = await (0, _install_index_templates.installSyntheticsIndexTemplates)(this.server);
        this.indexTemplateInstalling = false;
        if (installedPackage.name === 'synthetics' && installedPackage.install_status === 'installed') {
          this.logger.debug('Installed synthetics index templates');
          this.indexTemplateExists = true;
        } else if (installedPackage.name === 'synthetics' && installedPackage.install_status === 'install_failed') {
          const e = new IndexTemplateInstallationError();
          this.logger.error(e.message, {
            error: e
          });
          this.indexTemplateExists = false;
        }
      }
    } catch (e) {
      this.logger.error(new IndexTemplateInstallationError().message, {
        error: e
      });
      this.indexTemplateInstalling = false;
    }
  }
  async registerServiceLocations() {
    const service = this;
    try {
      const result = await (0, _get_service_locations.getServiceLocations)(service.server);
      service.throttling = result.throttling;
      service.locations = result.locations;
      service.apiClient.locations = result.locations;
      this.logger.debug(`Fetched ${service.locations.map(loc => loc.id).join(',')} Synthetics service locations from manifest: ${this.config.manifestUrl}`);
    } catch (error) {
      this.logger.error(`Error registering service locations, Error: ${error.message}`, {
        error
      });
    }
  }
  registerSyncTask(taskManager) {
    var _this$config$syncInte;
    const service = this;
    const interval = (_this$config$syncInte = this.config.syncInterval) !== null && _this$config$syncInte !== void 0 ? _this$config$syncInte : SYNTHETICS_SERVICE_SYNC_INTERVAL_DEFAULT;
    taskManager.registerTaskDefinitions({
      [SYNTHETICS_SERVICE_SYNC_MONITORS_TASK_TYPE]: {
        title: 'Synthetics Service - Sync Saved Monitors',
        description: 'This task periodically pushes saved monitors to Synthetics Service.',
        timeout: '2m',
        maxAttempts: 3,
        createTaskRunner: ({
          taskInstance
        }) => {
          return {
            // Perform the work of the task. The return value should fit the TaskResult interface.
            async run() {
              const {
                state
              } = taskInstance;
              service.logger.debug(`Running synthetics monitors sync task.`);
              service.checkMissingSchedule(state);
              try {
                await service.registerServiceLocations();
                const {
                  allowed,
                  signupUrl
                } = await service.apiClient.checkAccountAccessStatus();
                service.isAllowed = allowed;
                service.signupUrl = signupUrl;
                if (service.isAllowed && service.config.manifestUrl) {
                  void service.setupIndexTemplates();
                  await service.pushConfigs(_constants.ALL_SPACES_ID);
                } else {
                  if (!service.isAllowed) {
                    service.logger.debug('User is not allowed to access Synthetics service.');
                  }
                }
              } catch (e) {
                (0, _monitor_upgrade_sender.sendErrorTelemetryEvents)(service.logger, service.server.telemetry, {
                  reason: 'Failed to run scheduled sync task',
                  message: e === null || e === void 0 ? void 0 : e.message,
                  type: 'runTaskError',
                  code: e === null || e === void 0 ? void 0 : e.code,
                  status: e.status,
                  stackVersion: service.server.stackVersion
                });
                service.logger.error(e);
              }
              return {
                state,
                schedule: {
                  interval
                }
              };
            },
            async cancel() {
              var _service$logger;
              (_service$logger = service.logger) === null || _service$logger === void 0 ? void 0 : _service$logger.warn(`Task ${SYNTHETICS_SERVICE_SYNC_MONITORS_TASK_ID} timed out`);
            }
          };
        }
      }
    });
  }
  async scheduleSyncTask(taskManager) {
    var _this$config$syncInte2;
    const interval = (_this$config$syncInte2 = this.config.syncInterval) !== null && _this$config$syncInte2 !== void 0 ? _this$config$syncInte2 : SYNTHETICS_SERVICE_SYNC_INTERVAL_DEFAULT;
    try {
      var _this$logger, _taskInstance$schedul;
      const taskInstance = await taskManager.ensureScheduled({
        id: SYNTHETICS_SERVICE_SYNC_MONITORS_TASK_ID,
        taskType: SYNTHETICS_SERVICE_SYNC_MONITORS_TASK_TYPE,
        schedule: {
          interval
        },
        params: {},
        state: {},
        scope: ['uptime']
      });
      (_this$logger = this.logger) === null || _this$logger === void 0 ? void 0 : _this$logger.info(`Task ${SYNTHETICS_SERVICE_SYNC_MONITORS_TASK_ID} scheduled with interval ${(_taskInstance$schedul = taskInstance.schedule) === null || _taskInstance$schedul === void 0 ? void 0 : _taskInstance$schedul.interval}.`);
      return taskInstance;
    } catch (e) {
      var _e$message, _this$logger2, _this$logger3;
      (0, _monitor_upgrade_sender.sendErrorTelemetryEvents)(this.logger, this.server.telemetry, {
        reason: 'Failed to schedule sync task',
        message: (_e$message = e === null || e === void 0 ? void 0 : e.message) !== null && _e$message !== void 0 ? _e$message : e,
        type: 'scheduleTaskError',
        code: e === null || e === void 0 ? void 0 : e.code,
        status: e.status,
        stackVersion: this.server.stackVersion
      });
      (_this$logger2 = this.logger) === null || _this$logger2 === void 0 ? void 0 : _this$logger2.error(e);
      (_this$logger3 = this.logger) === null || _this$logger3 === void 0 ? void 0 : _this$logger3.error(`Error running synthetics syncs task: ${SYNTHETICS_SERVICE_SYNC_MONITORS_TASK_ID}, ${e === null || e === void 0 ? void 0 : e.message}`);
      return null;
    }
  }
  async getLicense() {
    var _license, _license2;
    this.esClient = this.getESClient();
    let license;
    if (this.esClient === undefined || this.esClient === null) {
      throw Error('Cannot sync monitors with the Synthetics service. Elasticsearch client is unavailable: cannot retrieve license information');
    }
    try {
      var _await$this$esClient$;
      license = (_await$this$esClient$ = await this.esClient.license.get()) === null || _await$this$esClient$ === void 0 ? void 0 : _await$this$esClient$.license;
    } catch (e) {
      throw new Error(`Cannot sync monitors with the Synthetics service. Unable to determine license level: ${e}`);
    }
    if (((_license = license) === null || _license === void 0 ? void 0 : _license.status) === 'expired') {
      throw new Error('Cannot sync monitors with the Synthetics service. License is expired.');
    }
    if (!((_license2 = license) !== null && _license2 !== void 0 && _license2.type)) {
      throw new Error('Cannot sync monitors with the Synthetics service. Unable to determine license level.');
    }
    return license;
  }
  async getSOClientFinder({
    pageSize
  }) {
    const encryptedClient = this.server.encryptedSavedObjects.getClient();
    return await encryptedClient.createPointInTimeFinderDecryptedAsInternalUser({
      type: [_saved_objects.legacySyntheticsMonitorTypeSingle, _saved_objects.syntheticsMonitorSavedObjectType],
      perPage: pageSize,
      namespaces: [_constants.ALL_SPACES_ID]
    });
  }
  getESClient() {
    var _this$server$coreStar;
    if (!this.server.coreStart) {
      return;
    }
    return (_this$server$coreStar = this.server.coreStart) === null || _this$server$coreStar === void 0 ? void 0 : _this$server$coreStar.elasticsearch.client.asInternalUser;
  }
  async getOutput({
    inspect
  } = {
    inspect: false
  }) {
    const {
      apiKey,
      isValid
    } = await (0, _get_api_key.getAPIKeyForSyntheticsService)({
      server: this.server
    });
    // do not check for api key validity if inspecting
    if (!isValid && !inspect) {
      this.server.logger.debug('API key is not valid. Cannot push monitor configuration to synthetics public testing locations');
      this.invalidApiKeyError = true;
      return null;
    }
    return {
      hosts: this.esHosts,
      api_key: `${apiKey === null || apiKey === void 0 ? void 0 : apiKey.id}:${apiKey === null || apiKey === void 0 ? void 0 : apiKey.apiKey}`
    };
  }
  async inspectConfig(config, mws) {
    if (!config || (0, _lodash.isEmpty)(config)) {
      return null;
    }
    const monitors = this.formatConfigs(config, mws);
    const license = await this.getLicense();
    const output = await this.getOutput({
      inspect: true
    });
    if (output) {
      return await this.apiClient.inspect({
        monitors,
        output,
        license
      });
    }
    return null;
  }
  async addConfigs(configs, mws) {
    try {
      if (configs.length === 0 || !this.isAllowed) {
        return;
      }
      const monitors = this.formatConfigs(configs, mws);
      const license = await this.getLicense();
      const output = await this.getOutput();
      if (output) {
        this.logger.debug(`1 monitor will be pushed to synthetics service.`);
        this.apiClient.post({
          monitors,
          output,
          license
        }).then(res => {
          this.syncErrors = res;
        }).catch(e => {
          this.logger.error(e);
        });
      }
      return this.syncErrors;
    } catch (e) {
      this.logger.error(e);
    }
  }
  async editConfig(monitorConfig, isEdit = true, mws) {
    try {
      if (monitorConfig.length === 0 || !this.isAllowed) {
        return;
      }
      const license = await this.getLicense();
      const monitors = this.formatConfigs(monitorConfig, mws);
      const output = await this.getOutput();
      if (output) {
        const data = {
          monitors,
          output,
          isEdit,
          license
        };
        this.syncErrors = await this.apiClient.put(data);
      }
      return this.syncErrors;
    } catch (e) {
      this.logger.error(e);
    }
  }
  async pushConfigs(spaceId) {
    const license = await this.getLicense();
    const service = this;
    const PER_PAGE = 250;
    service.syncErrors = [];
    let output = null;
    const paramsBySpace = await this.getSyntheticsParams();
    const maintenanceWindows = await this.getMaintenanceWindows(spaceId);
    const finder = await this.getSOClientFinder({
      pageSize: PER_PAGE
    });
    const bucketsByLocation = {};
    this.locations.forEach(location => {
      bucketsByLocation[location.id] = [];
    });
    const syncAllLocations = async (perBucket = 0) => {
      await (0, _pMap.default)(this.locations, async location => {
        if (bucketsByLocation[location.id].length > perBucket && output) {
          var _this$syncErrors;
          const locMonitors = bucketsByLocation[location.id].splice(0, PER_PAGE);
          this.logger.debug(`${locMonitors.length} monitors will be pushed to synthetics service for location ${location.id}.`);
          const syncErrors = await this.apiClient.syncMonitors({
            monitors: locMonitors,
            output,
            license,
            location
          });
          this.syncErrors = [...((_this$syncErrors = this.syncErrors) !== null && _this$syncErrors !== void 0 ? _this$syncErrors : []), ...(syncErrors !== null && syncErrors !== void 0 ? syncErrors : [])];
        }
      }, {
        stopOnError: false
      });
    };
    for await (const result of finder.find()) {
      if (result.saved_objects.length > 0) {
        try {
          if (!output) {
            output = await this.getOutput();
            if (!output) {
              (0, _monitor_upgrade_sender.sendErrorTelemetryEvents)(service.logger, service.server.telemetry, {
                reason: 'API key is not valid.',
                message: 'Failed to push configs. API key is not valid.',
                type: 'invalidApiKey',
                stackVersion: service.server.stackVersion
              });
              return;
            }
          }
          const monitors = result.saved_objects.filter(({
            error
          }) => !error);
          const formattedConfigs = this.normalizeConfigs(monitors, paramsBySpace, maintenanceWindows);
          this.logger.debug(`${formattedConfigs.length} monitors will be pushed to synthetics service.`);
          formattedConfigs.forEach(monitor => {
            monitor.locations.forEach(location => {
              if (location.isServiceManaged) {
                var _bucketsByLocation$lo;
                (_bucketsByLocation$lo = bucketsByLocation[location.id]) === null || _bucketsByLocation$lo === void 0 ? void 0 : _bucketsByLocation$lo.push(monitor);
              }
            });
          });
          await syncAllLocations(PER_PAGE);
        } catch (error) {
          this.logger.error(`Failed to run Synthetics sync task, Error: ${error.message}`, {
            error
          });
          (0, _monitor_upgrade_sender.sendErrorTelemetryEvents)(service.logger, service.server.telemetry, {
            reason: 'Failed to push configs to service',
            message: error === null || error === void 0 ? void 0 : error.message,
            type: 'pushConfigsError',
            code: error === null || error === void 0 ? void 0 : error.code,
            status: error.status,
            stackVersion: service.server.stackVersion
          });
        }
      }
    }

    // execute the remaining monitors
    await syncAllLocations();
    await finder.close();
  }
  async runOnceConfigs(configs) {
    if (!configs) {
      return;
    }
    const monitors = this.formatConfigs(configs, []);
    if (monitors.length === 0) {
      return;
    }
    const license = await this.getLicense();
    const output = await this.getOutput();
    if (!output) {
      return;
    }
    try {
      return await this.apiClient.runOnce({
        monitors,
        output,
        license
      });
    } catch (e) {
      this.logger.error(e);
      throw e;
    }
  }
  async deleteConfigs(configs) {
    try {
      if (configs.length === 0) {
        return;
      }
      const license = await this.getLicense();
      const hasPublicLocations = configs.some(config => config.monitor.locations.some(({
        isServiceManaged
      }) => isServiceManaged));
      if (hasPublicLocations) {
        const output = await this.getOutput();
        if (!output) {
          return;
        }
        const data = {
          output,
          monitors: this.formatConfigs(configs, []),
          license
        };
        return await this.apiClient.delete(data);
      }
    } catch (e) {
      this.server.logger.error(e);
    }
  }
  async deleteAllConfigs() {
    const license = await this.getLicense();
    const finder = await this.getSOClientFinder({
      pageSize: 100
    });
    const output = await this.getOutput();
    if (!output) {
      return;
    }
    for await (const result of finder.find()) {
      const monitors = this.normalizeConfigs(result.saved_objects, {}, []);
      const hasPublicLocations = monitors.some(config => config.locations.some(({
        isServiceManaged
      }) => isServiceManaged));
      if (hasPublicLocations) {
        const data = {
          output,
          monitors,
          license
        };
        return await this.apiClient.delete(data);
      }
    }
  }
  async getSyntheticsParams({
    spaceId,
    hideParams = false,
    canSave = true
  } = {}) {
    if (!canSave) {
      return Object.create(null);
    }
    const encryptedClient = this.server.encryptedSavedObjects.getClient();
    const paramsBySpace = Object.create(null);
    const finder = await encryptedClient.createPointInTimeFinderDecryptedAsInternalUser({
      type: _saved_objects.syntheticsParamType,
      perPage: 1000,
      namespaces: spaceId ? [spaceId] : [_constants.ALL_SPACES_ID]
    });
    for await (const response of finder.find()) {
      response.saved_objects.forEach(param => {
        var _param$namespaces;
        (_param$namespaces = param.namespaces) === null || _param$namespaces === void 0 ? void 0 : _param$namespaces.forEach(namespace => {
          if (!paramsBySpace[namespace]) {
            paramsBySpace[namespace] = Object.create(null);
          }
          paramsBySpace[namespace][param.attributes.key] = hideParams ? '"*******"' : param.attributes.value;
        });
      });
    }

    // no need to wait here
    finder.close().catch(() => {});
    if (paramsBySpace[_constants.ALL_SPACES_ID]) {
      Object.keys(paramsBySpace).forEach(space => {
        if (space !== _constants.ALL_SPACES_ID) {
          var _paramsBySpace$space, _paramsBySpace$ALL_SP;
          paramsBySpace[space] = {
            ...((_paramsBySpace$space = paramsBySpace[space]) !== null && _paramsBySpace$space !== void 0 ? _paramsBySpace$space : {}),
            ...((_paramsBySpace$ALL_SP = paramsBySpace[_constants.ALL_SPACES_ID]) !== null && _paramsBySpace$ALL_SP !== void 0 ? _paramsBySpace$ALL_SP : {})
          };
        }
      });
      if (spaceId) {
        var _paramsBySpace$spaceI, _paramsBySpace$ALL_SP2;
        paramsBySpace[spaceId] = {
          ...((_paramsBySpace$spaceI = paramsBySpace === null || paramsBySpace === void 0 ? void 0 : paramsBySpace[spaceId]) !== null && _paramsBySpace$spaceI !== void 0 ? _paramsBySpace$spaceI : {}),
          ...((_paramsBySpace$ALL_SP2 = paramsBySpace === null || paramsBySpace === void 0 ? void 0 : paramsBySpace[_constants.ALL_SPACES_ID]) !== null && _paramsBySpace$ALL_SP2 !== void 0 ? _paramsBySpace$ALL_SP2 : {})
        };
      }
    }
    return paramsBySpace;
  }
  async getMaintenanceWindows(spaceId) {
    const {
      savedObjects
    } = this.server.coreStart;
    const soClient = savedObjects.createInternalRepository([_common2.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE]);
    const maintenanceWindowClient = new _maintenance_window_client.MaintenanceWindowClient({
      savedObjectsClient: soClient,
      getUserName: async () => '',
      uiSettings: this.server.coreStart.uiSettings.asScopedToClient(soClient),
      logger: this.logger
    });
    const mws = await maintenanceWindowClient.find({
      page: 0,
      perPage: 1000,
      namespaces: [spaceId]
    });
    return mws.data;
  }
  formatConfigs(configData, mws) {
    const configDataList = Array.isArray(configData) ? configData : [configData];
    return configDataList.map(config => {
      const {
        str: paramsString,
        params
      } = (0, _format_configs.mixParamsWithGlobalParams)(config.params, config.monitor);
      const asHeartbeatConfig = (0, _format_configs.formatHeartbeatRequest)(config, paramsString);
      return (0, _format_configs.formatMonitorConfigFields)(Object.keys(asHeartbeatConfig), asHeartbeatConfig, this.logger, params !== null && params !== void 0 ? params : {}, mws);
    });
  }
  normalizeConfigs(monitors, paramsBySpace, mws) {
    const configDataList = (monitors !== null && monitors !== void 0 ? monitors : []).map(monitor => {
      var _monitor$namespaces$, _monitor$namespaces, _paramsBySpace$monito, _paramsBySpace$ALL_SP3;
      const attributes = monitor.attributes;
      const monitorSpace = (_monitor$namespaces$ = (_monitor$namespaces = monitor.namespaces) === null || _monitor$namespaces === void 0 ? void 0 : _monitor$namespaces[0]) !== null && _monitor$namespaces$ !== void 0 ? _monitor$namespaces$ : _common.DEFAULT_SPACE_ID;
      const params = (_paramsBySpace$monito = paramsBySpace[monitorSpace]) !== null && _paramsBySpace$monito !== void 0 ? _paramsBySpace$monito : {};
      return {
        params: {
          ...params,
          ...((_paramsBySpace$ALL_SP3 = paramsBySpace === null || paramsBySpace === void 0 ? void 0 : paramsBySpace[_constants.ALL_SPACES_ID]) !== null && _paramsBySpace$ALL_SP3 !== void 0 ? _paramsBySpace$ALL_SP3 : {})
        },
        monitor: (0, _secrets.normalizeSecrets)(monitor).attributes,
        configId: monitor.id,
        heartbeatId: attributes[_runtime_types.ConfigKey.MONITOR_QUERY_ID],
        spaceId: monitorSpace
      };
    });
    return this.formatConfigs(configDataList, mws);
  }
  checkMissingSchedule(state) {
    try {
      const lastRunAt = state.lastRunAt;
      const current = (0, _moment.default)();
      if (lastRunAt) {
        var _this$config$syncInte3;
        // log if it has missed last schedule
        const diff = (0, _moment.default)(current).diff(lastRunAt, 'minutes');
        const syncInterval = Number(((_this$config$syncInte3 = this.config.syncInterval) !== null && _this$config$syncInte3 !== void 0 ? _this$config$syncInte3 : '5m').split('m')[0]) + 5;
        if (diff > syncInterval) {
          const message = `Synthetics monitor sync task has missed its schedule, it last ran ${diff} minutes ago.`;
          this.logger.warn(message);
          (0, _monitor_upgrade_sender.sendErrorTelemetryEvents)(this.logger, this.server.telemetry, {
            message,
            reason: 'Failed to run synthetics sync task on schedule',
            type: 'syncTaskMissedSchedule',
            stackVersion: this.server.stackVersion
          });
        }
        this.logger.debug(`Synthetics monitor sync task last ran ${diff} minutes ago.`);
      }
      state.lastRunAt = current.toISOString();
    } catch (e) {
      this.logger.error(e);
    }
  }
}
exports.SyntheticsService = SyntheticsService;
class IndexTemplateInstallationError extends Error {
  constructor() {
    super();
    this.message = 'Failed to install synthetics index templates.';
    this.name = 'IndexTemplateInstallationError';
  }
}