"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.editSyntheticsParamsRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/core/server");
var _lodash = require("lodash");
var _common = require("../../common");
var _saved_objects = require("../../../../common/types/saved_objects");
var _constants = require("../../../../common/constants");
var _sync_global_params_task = require("../../../tasks/sync_global_params_task");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const RequestParamsSchema = _configSchema.schema.object({
  id: _configSchema.schema.string()
});
const editSyntheticsParamsRoute = () => ({
  method: 'PUT',
  path: _constants.SYNTHETICS_API_URLS.PARAMS + '/{id}',
  validate: {},
  validation: {
    request: {
      params: RequestParamsSchema,
      body: _configSchema.schema.object({
        key: _configSchema.schema.maybe(_configSchema.schema.string({
          minLength: 1
        })),
        value: _configSchema.schema.maybe(_configSchema.schema.string({
          minLength: 1
        })),
        description: _configSchema.schema.maybe(_configSchema.schema.string()),
        tags: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
      })
    }
  },
  handler: async routeContext => {
    const {
      savedObjectsClient,
      request,
      response,
      spaceId,
      server
    } = routeContext;
    const {
      invalidResponse
    } = await (0, _common.validateRouteSpaceName)(routeContext);
    if (invalidResponse) return invalidResponse;
    const {
      id: paramId
    } = request.params;
    const data = request.body;
    if ((0, _lodash.isEmpty)(data)) {
      return response.badRequest({
        body: {
          message: 'Request body cannot be empty'
        }
      });
    }
    const encryptedSavedObjectsClient = server.encryptedSavedObjects.getClient();
    try {
      const existingParam = await encryptedSavedObjectsClient.getDecryptedAsInternalUser(_saved_objects.syntheticsParamType, paramId, {
        namespace: spaceId
      });
      const newParam = {
        ...existingParam.attributes,
        ...data
      };

      // value from data since we aren't using encrypted client
      const {
        value
      } = existingParam.attributes;
      const {
        id: responseId,
        attributes: {
          key,
          tags,
          description
        },
        namespaces
      } = await savedObjectsClient.update(_saved_objects.syntheticsParamType, paramId, newParam);
      await (0, _sync_global_params_task.asyncGlobalParamsPropagation)({
        server,
        paramsSpacesToSync: existingParam.namespaces || [spaceId]
      });
      return {
        id: responseId,
        key,
        tags,
        description,
        namespaces,
        value
      };
    } catch (getErr) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(getErr)) {
        return response.notFound({
          body: {
            message: 'Param not found'
          }
        });
      }
    }
  }
});
exports.editSyntheticsParamsRoute = editSyntheticsParamsRoute;