"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.enablementRoutes = exports.enableStreamsRoute = exports.disableStreamsRoute = void 0;
var _zod = require("@kbn/zod");
var _boom = require("@hapi/boom");
var _constants = require("../../../../common/constants");
var _name_taken_error = require("../../../lib/streams/errors/name_taken_error");
var _create_server_route = require("../../create_server_route");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const enableStreamsRoute = exports.enableStreamsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /api/streams/_enable 2023-10-31',
  params: _zod.z.object({}),
  options: {
    access: 'public',
    summary: 'Enable streams',
    description: 'Enables wired streams',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  handler: async ({
    request,
    getScopedClients
  }) => {
    const {
      streamsClient
    } = await getScopedClients({
      request
    });
    try {
      return await streamsClient.enableStreams();
    } catch (error) {
      if (error instanceof _name_taken_error.NameTakenError) {
        throw (0, _boom.conflict)(`Cannot enable Streams, failed to create root stream: ${error.message}`);
      }
      throw error;
    }
  }
});
const disableStreamsRoute = exports.disableStreamsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /api/streams/_disable 2023-10-31',
  params: _zod.z.object({}),
  options: {
    access: 'public',
    summary: 'Disable streams',
    description: 'Disables wired streams and deletes all existing stream definitions. The data of wired streams is deleted, but the data of classic streams is preserved.',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  handler: async ({
    request,
    getScopedClients
  }) => {
    const {
      streamsClient
    } = await getScopedClients({
      request
    });
    return await streamsClient.disableStreams();
  }
});
const enablementRoutes = exports.enablementRoutes = {
  ...enableStreamsRoute,
  ...disableStreamsRoute
};