"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.queryRoutes = void 0;
var _streamsSchema = require("@kbn/streams-schema");
var _zod = require("@kbn/zod");
var _constants = require("../../../common/constants");
var _query_not_found_error = require("../../lib/streams/errors/query_not_found_error");
var _create_server_route = require("../create_server_route");
var _assert_enterprise_license = require("../utils/assert_enterprise_license");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const listQueriesRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /api/streams/{name}/queries 2023-10-31',
  options: {
    access: 'public',
    summary: 'Get stream queries',
    description: 'Fetches all queries linked to a stream that are visible to the current user in the current space.',
    availability: {
      stability: 'experimental'
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    })
  }),
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  async handler({
    params,
    request,
    getScopedClients
  }) {
    const {
      assetClient,
      streamsClient,
      licensing
    } = await getScopedClients({
      request
    });
    await (0, _assert_enterprise_license.assertEnterpriseLicense)(licensing);
    await streamsClient.ensureStream(params.path.name);
    const {
      path: {
        name: streamName
      }
    } = params;
    const {
      [streamName]: queryAssets
    } = await assetClient.getAssetLinks([streamName], ['query']);
    return {
      queries: queryAssets.map(queryAsset => queryAsset.query)
    };
  }
});
const upsertQueryRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'PUT /api/streams/{name}/queries/{queryId} 2023-10-31',
  options: {
    access: 'public',
    summary: 'Upsert a query to a stream',
    description: 'Adds a query to a stream. Noop if the query is already present on the stream.',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string(),
      queryId: _zod.z.string()
    }),
    body: _streamsSchema.upsertStreamQueryRequestSchema
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      streamsClient,
      queryClient,
      licensing
    } = await getScopedClients({
      request
    });
    const {
      path: {
        name: streamName,
        queryId
      },
      body
    } = params;
    await (0, _assert_enterprise_license.assertEnterpriseLicense)(licensing);
    await streamsClient.ensureStream(streamName);
    await queryClient.upsert(streamName, {
      id: queryId,
      title: body.title,
      feature: body.feature,
      kql: {
        query: body.kql.query
      }
    });
    return {
      acknowledged: true
    };
  }
});
const deleteQueryRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'DELETE /api/streams/{name}/queries/{queryId} 2023-10-31',
  options: {
    access: 'public',
    summary: 'Remove a query from a stream',
    description: 'Remove a query from a stream. Noop if the query is not found on the stream.',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string(),
      queryId: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      streamsClient,
      queryClient,
      licensing,
      assetClient
    } = await getScopedClients({
      request
    });
    await (0, _assert_enterprise_license.assertEnterpriseLicense)(licensing);
    const {
      path: {
        queryId,
        name: streamName
      }
    } = params;
    await streamsClient.ensureStream(streamName);
    const queryLink = await assetClient.bulkGetByIds(streamName, 'query', [queryId]);
    if (queryLink.length === 0) {
      throw new _query_not_found_error.QueryNotFoundError(`Query [${queryId}] not found in stream [${streamName}]`);
    }
    await queryClient.delete(streamName, queryId);
    return {
      acknowledged: true
    };
  }
});
const bulkQueriesRoute = (0, _create_server_route.createServerRoute)({
  endpoint: `POST /api/streams/{name}/queries/_bulk 2023-10-31`,
  options: {
    access: 'public',
    summary: 'Bulk update queries',
    description: 'Bulk update queries of a stream. Can add new queries and delete existing ones.',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    body: _zod.z.object({
      operations: _zod.z.array(_zod.z.union([_zod.z.object({
        index: _streamsSchema.streamQuerySchema
      }), _zod.z.object({
        delete: _zod.z.object({
          id: _zod.z.string()
        })
      })]))
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      streamsClient,
      queryClient,
      licensing
    } = await getScopedClients({
      request
    });
    await (0, _assert_enterprise_license.assertEnterpriseLicense)(licensing);
    const {
      path: {
        name: streamName
      },
      body: {
        operations
      }
    } = params;
    await streamsClient.ensureStream(streamName);
    await queryClient.bulk(streamName, operations);
    return {
      acknowledged: true
    };
  }
});
const queryRoutes = exports.queryRoutes = {
  ...listQueriesRoute,
  ...upsertQueryRoute,
  ...deleteQueryRoute,
  ...bulkQueriesRoute
};