"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processingDateSuggestionsSchema = exports.handleProcessingDateSuggestions = void 0;
var _lodash = require("lodash");
var _zod = require("@kbn/zod");
var _zodHelpers = require("@kbn/zod-helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const processingDateSuggestionsSchema = exports.processingDateSuggestionsSchema = _zod.z.object({
  path: _zod.z.object({
    name: _zod.z.string()
  }),
  body: _zod.z.object({
    dates: _zod.z.array(_zod.z.unknown()).nonempty()
  })
});
const handleProcessingDateSuggestions = async ({
  params,
  scopedClusterClient
}) => {
  const dates = parseDatesInput(params.body.dates);
  /**
   * Run structure detection against sample dates.
   * The `findMessageStructure` API is used to detect the structure of the date strings.
   * One limitation of this API is that it uses a single detection pattern that matches the first message, so upcoming messages with a much different structure would cause a generic failure.
   * To work around this, we have a series of attempts:
   * - Run a single detection. If successful, we collect the result.
   * - If the first detection fails, we run the detection against all sample dates individually and collect the unique formats.
   * - If the second detection fails, we return an empty response.
   */
  let formats = [];

  /**
   * 1. Run a single detection. If successful, we collect the result.
   */
  try {
    formats = await attemptSingleDetection({
      scopedClusterClient,
      dates
    });
    return {
      formats
    };
  } catch {
    // Noop, continue to the next step
    // We gracefully handle detection failures by returning an empty response.
  }

  /**
   * 2. Run the detection against all sample dates individually and collect the unique formats.
   */
  try {
    formats = await attemptParallelDetection({
      scopedClusterClient,
      dates
    });
  } catch {
    // Noop, continue to the next step
    // We gracefully handle detection failures by returning an empty response.
  }
  return {
    formats
  };
};
exports.handleProcessingDateSuggestions = handleProcessingDateSuggestions;
function parseDatesInput(dates) {
  const areValidDates = _zod.z.array(_zod.z.union([_zodHelpers.NonEmptyString, _zod.z.number()])).nonempty().safeParse(dates).success;
  if (!areValidDates) {
    throw new Error('Dates input must be non-empty string or number values.');
  }
  return dates.map(String);
}
const attemptSingleDetection = async ({
  scopedClusterClient,
  dates
}) => {
  var _textStructureRespons;
  const textStructureResponse = await detectDatesStructure({
    scopedClusterClient,
    dates
  });
  const formats = (_textStructureRespons = textStructureResponse.java_timestamp_formats) !== null && _textStructureRespons !== void 0 ? _textStructureRespons : [];
  return formats;
};
const attemptParallelDetection = async ({
  scopedClusterClient,
  dates
}) => {
  const textStructureResponses = await Promise.allSettled(dates.map(date => detectDatesStructure({
    scopedClusterClient,
    dates: [date]
  })));
  const formats = (0, _lodash.uniq)(textStructureResponses.filter(isFulfilled) // Filter out any rejected promises
  .flatMap(textStructureResponse => textStructureResponse.value.java_timestamp_formats) // Flatten the array of detected formats
  .filter(_lodash.isString) // Filter out any undefined values
  );
  return formats;
};
function detectDatesStructure({
  scopedClusterClient,
  dates
}) {
  return scopedClusterClient.asInternalUser.textStructure.findMessageStructure({
    messages: dates,
    format: 'semi_structured_text',
    ecs_compatibility: 'v1'
  });
}
function isFulfilled(item) {
  return item.status === 'fulfilled';
}