"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.dashboardRoutes = void 0;
var _zod = require("@kbn/zod");
var _boom = require("@hapi/boom");
var _constants = require("../../../common/constants");
var _create_server_route = require("../create_server_route");
var _fields = require("../../lib/streams/assets/fields");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function sanitizeDashboardAsset(asset) {
  return {
    id: asset[_fields.ASSET_ID],
    title: asset.title,
    tags: asset.tags
  };
}
const listDashboardsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /api/streams/{name}/dashboards 2023-10-31',
  options: {
    access: 'public',
    summary: 'Get stream dashboards',
    description: 'Fetches all dashboards linked to a stream that are visible to the current user in the current space.',
    availability: {
      stability: 'experimental'
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    })
  }),
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  async handler({
    params,
    request,
    getScopedClients
  }) {
    const {
      assetClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    await streamsClient.ensureStream(params.path.name);
    const {
      path: {
        name: streamName
      }
    } = params;
    function isDashboard(asset) {
      return asset[_fields.ASSET_TYPE] === 'dashboard';
    }
    return {
      dashboards: (await assetClient.getAssets(streamName)).filter(isDashboard).map(sanitizeDashboardAsset)
    };
  }
});
const linkDashboardRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'PUT /api/streams/{name}/dashboards/{dashboardId} 2023-10-31',
  options: {
    access: 'public',
    summary: 'Link a dashboard to a stream',
    description: 'Links a dashboard to a stream. Noop if the dashboard is already linked to the stream.',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string(),
      dashboardId: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      assetClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const {
      path: {
        dashboardId,
        name: streamName
      }
    } = params;
    await streamsClient.ensureStream(streamName);
    await assetClient.linkAsset(streamName, {
      [_fields.ASSET_TYPE]: 'dashboard',
      [_fields.ASSET_ID]: dashboardId
    });
    return {
      acknowledged: true
    };
  }
});
const unlinkDashboardRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'DELETE /api/streams/{name}/dashboards/{dashboardId} 2023-10-31',
  options: {
    access: 'public',
    summary: 'Unlink a dashboard from a stream',
    description: 'Unlinks a dashboard from a stream. Noop if the dashboard is not linked to the stream.',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string(),
      dashboardId: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      assetClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    await streamsClient.ensureStream(params.path.name);
    const {
      path: {
        dashboardId,
        name: streamName
      }
    } = params;
    await assetClient.unlinkAsset(streamName, {
      [_fields.ASSET_ID]: dashboardId,
      [_fields.ASSET_TYPE]: 'dashboard'
    });
    return {
      acknowledged: true
    };
  }
});
const suggestDashboardsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /internal/streams/{name}/dashboards/_suggestions',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    query: _zod.z.object({
      query: _zod.z.string()
    }),
    body: _zod.z.object({
      tags: _zod.z.optional(_zod.z.array(_zod.z.string()))
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      assetClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    await streamsClient.ensureStream(params.path.name);
    const {
      query: {
        query
      },
      body: {
        tags
      }
    } = params;
    const suggestions = (await assetClient.getSuggestions({
      assetTypes: ['dashboard'],
      query,
      tags
    })).assets.map(asset => {
      return sanitizeDashboardAsset(asset);
    });
    return {
      suggestions
    };
  }
});
const dashboardSchema = _zod.z.object({
  id: _zod.z.string()
});
const bulkDashboardsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: `POST /api/streams/{name}/dashboards/_bulk 2023-10-31`,
  options: {
    access: 'public',
    summary: 'Bulk update dashboards',
    description: 'Bulk update dashboards linked to a stream. Can link new dashboards and delete existing ones.',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    body: _zod.z.object({
      operations: _zod.z.array(_zod.z.union([_zod.z.object({
        index: dashboardSchema
      }), _zod.z.object({
        delete: dashboardSchema
      })]))
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients,
    logger
  }) => {
    const {
      assetClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const {
      path: {
        name: streamName
      },
      body: {
        operations
      }
    } = params;
    await streamsClient.ensureStream(streamName);
    const result = await assetClient.bulk(streamName, operations.map(operation => {
      if ('index' in operation) {
        return {
          index: {
            asset: {
              [_fields.ASSET_TYPE]: 'dashboard',
              [_fields.ASSET_ID]: operation.index.id
            }
          }
        };
      }
      return {
        delete: {
          asset: {
            [_fields.ASSET_TYPE]: 'dashboard',
            [_fields.ASSET_ID]: operation.delete.id
          }
        }
      };
    }));
    if (result.errors) {
      logger.error(`Error indexing some items`);
      throw (0, _boom.internal)(`Could not index all items`, {
        errors: result.errors
      });
    }
    return {
      acknowledged: true
    };
  }
});
const dashboardRoutes = exports.dashboardRoutes = {
  ...listDashboardsRoute,
  ...linkDashboardRoute,
  ...unlinkDashboardRoute,
  ...suggestDashboardsRoute,
  ...bulkDashboardsRoute
};