"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createStreamsGlobalSearchResultProvider = createStreamsGlobalSearchResultProvider;
var _rxjs = require("rxjs");
var _streamsSchema = require("@kbn/streams-schema");
var _managementSettingsIds = require("@kbn/management-settings-ids");
var _streams_storage_client = require("./storage/streams_storage_client");
var _stream_crud = require("./stream_crud");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const streamTypes = ['classic stream', 'wired stream', 'group stream', 'stream'];
function createStreamsGlobalSearchResultProvider(core, logger) {
  return {
    id: 'streams',
    getSearchableTypes: () => streamTypes,
    find: ({
      term = '',
      types = []
    }, {
      aborted$,
      maxResults,
      client
    }) => {
      if (!client) {
        return (0, _rxjs.from)([]);
      }
      const storageClient = (0, _streams_storage_client.createStreamsStorageClient)(client.asInternalUser, logger);
      return (0, _rxjs.from)(findStreams({
        term,
        types,
        maxResults,
        storageClient,
        client,
        core
      })).pipe((0, _rxjs.takeUntil)(aborted$));
    }
  };
}
async function findStreams({
  term,
  types,
  maxResults,
  storageClient,
  client,
  core
}) {
  const [coreStart] = await core.getStartServices();
  const soClient = coreStart.savedObjects.getUnsafeInternalClient();
  const uiSettingsClient = coreStart.uiSettings.asScopedToClient(soClient);
  const groupStreamsEnabled = await uiSettingsClient.get(_managementSettingsIds.OBSERVABILITY_STREAMS_ENABLE_GROUP_STREAMS);

  // This does NOT included unmanaged Classic streams
  const searchResponse = await storageClient.search({
    size: maxResults,
    track_total_hits: false,
    query: {
      bool: {
        should: [{
          wildcard: {
            name: {
              value: `*${term}*`,
              case_insensitive: true
            }
          }
        }, {
          match: {
            description: term
          }
        }],
        minimum_should_match: 1
      }
    }
  });
  const privileges = await (0, _stream_crud.checkAccessBulk)({
    names: searchResponse.hits.hits.filter(hit => !_streamsSchema.Streams.GroupStream.Definition.is(hit._source)).map(hit => hit._source.name),
    scopedClusterClient: client
  });
  const hitsWithAccess = searchResponse.hits.hits.filter(hit => {
    var _privileges$hit$_sour;
    if (_streamsSchema.Streams.GroupStream.Definition.is(hit._source)) return groupStreamsEnabled;
    return ((_privileges$hit$_sour = privileges[hit._source.name]) === null || _privileges$hit$_sour === void 0 ? void 0 : _privileges$hit$_sour.read) === true;
  });
  if (types.length === 0) {
    return hitsWithAccess.map(hit => toGlobalSearchProviderResult(hit._id, hit._source, term));
  }
  const relevantTypes = types.filter(type => streamTypes.includes(type));
  if (relevantTypes.length === 0) {
    return [];
  }
  if (relevantTypes.includes('stream')) {
    return hitsWithAccess.map(hit => toGlobalSearchProviderResult(hit._id, hit._source, term));
  }
  const includeClassicStream = relevantTypes.includes('classic stream');
  const includeWiredStream = relevantTypes.includes('wired stream');
  const includeGroupStream = relevantTypes.includes('group stream');
  const includeStream = ({
    _source
  }) => {
    return includeClassicStream && _streamsSchema.Streams.ClassicStream.Definition.is(_source) || includeWiredStream && _streamsSchema.Streams.WiredStream.Definition.is(_source) || includeGroupStream && _streamsSchema.Streams.GroupStream.Definition.is(_source);
  };
  return hitsWithAccess.filter(includeStream).map(hit => toGlobalSearchProviderResult(hit._id, hit._source, term));
}
function toGlobalSearchProviderResult(id, definition, term) {
  const type = _streamsSchema.Streams.ClassicStream.Definition.is(definition) ? 'Classic stream' : _streamsSchema.Streams.WiredStream.Definition.is(definition) ? 'Wired stream' : _streamsSchema.Streams.GroupStream.Definition.is(definition) ? 'Group stream' : 'Stream';
  const score = _streamsSchema.Streams.GroupStream.Definition.is(definition) ? boostGroupStreamScore(definition.name.toLowerCase(), term.toLowerCase()) : scoreStream(definition.name.toLowerCase(), term.toLowerCase());
  return {
    id,
    score,
    title: definition.name,
    type,
    url: `/app/streams/${definition.name}`
  };
}
function boostGroupStreamScore(name, searchTerm) {
  if (name === searchTerm) {
    return 100;
  } else if (name.startsWith(searchTerm)) {
    return 90;
  } else if (name.includes(searchTerm)) {
    return 80;
  }
  return 0;
}
function scoreStream(name, searchTerm) {
  if (name === searchTerm) {
    return 85;
  } else if (name.startsWith(searchTerm)) {
    return 75;
  } else if (name.includes(searchTerm)) {
    return 65;
  }
  return 0;
}