"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.readSignificantEventsFromAlertsIndices = readSignificantEventsFromAlertsIndices;
var _elasticsearch = require("@elastic/elasticsearch");
var _lodash = require("lodash");
var _query = require("../streams/assets/query/helpers/query");
var _security_error = require("../streams/errors/security_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function readSignificantEventsFromAlertsIndices(params, dependencies) {
  const {
    assetClient,
    scopedClusterClient
  } = dependencies;
  const {
    name,
    from,
    to,
    bucketSize
  } = params;
  const {
    [name]: queryLinks
  } = await assetClient.getAssetLinks([name], ['query']);
  if ((0, _lodash.isEmpty)(queryLinks)) {
    return [];
  }
  const queryLinkByRuleId = (0, _lodash.keyBy)(queryLinks, queryLink => (0, _query.getRuleIdFromQueryLink)(queryLink));
  const ruleIds = Object.keys(queryLinkByRuleId);
  const response = await scopedClusterClient.asCurrentUser.search({
    index: '.alerts-streams.alerts-default',
    query: {
      bool: {
        filter: [{
          range: {
            '@timestamp': {
              gte: from.toISOString(),
              lte: to.toISOString()
            }
          }
        }, {
          terms: {
            'kibana.alert.rule.uuid': ruleIds
          }
        }]
      }
    },
    aggs: {
      by_rule: {
        terms: {
          field: 'kibana.alert.rule.uuid',
          size: 10000
        },
        aggs: {
          occurrences: {
            date_histogram: {
              field: '@timestamp',
              fixed_interval: bucketSize,
              extended_bounds: {
                min: from.toISOString(),
                max: to.toISOString()
              }
            }
          },
          change_points: {
            // @ts-expect-error
            change_point: {
              buckets_path: 'occurrences>_count'
            }
          }
        }
      }
    }
  }).catch(err => {
    var _err$body, _err$body$error;
    const isResponseError = err instanceof _elasticsearch.errors.ResponseError;
    if (isResponseError && (err === null || err === void 0 ? void 0 : (_err$body = err.body) === null || _err$body === void 0 ? void 0 : (_err$body$error = _err$body.error) === null || _err$body$error === void 0 ? void 0 : _err$body$error.type) === 'security_exception') {
      throw new _security_error.SecurityError(`Cannot read significant events, insufficient privileges: ${err.message}`, {
        cause: err
      });
    }
    throw err;
  });
  if (!response.aggregations || !(0, _lodash.isArray)(response.aggregations.by_rule.buckets)) {
    return queryLinks.map(queryLink => ({
      id: queryLink.query.id,
      title: queryLink.query.title,
      kql: queryLink.query.kql,
      feature: queryLink.query.feature,
      occurrences: [],
      change_points: {
        type: {
          stationary: {
            p_value: 0,
            change_point: 0
          }
        }
      }
    }));
  }
  const significantEvents = response.aggregations.by_rule.buckets.map(bucket => {
    var _get;
    const ruleId = bucket.key;
    const queryLink = queryLinkByRuleId[ruleId];
    const occurrences = (0, _lodash.get)(bucket, 'occurrences.buckets');
    const changePoints = (_get = (0, _lodash.get)(bucket, 'change_points')) !== null && _get !== void 0 ? _get : {};
    return {
      id: queryLink.query.id,
      title: queryLink.query.title,
      kql: queryLink.query.kql,
      feature: queryLink.query.feature,
      occurrences: (0, _lodash.isArray)(occurrences) ? occurrences.map(occurrence => ({
        date: occurrence.key_as_string,
        count: occurrence.doc_count
      })) : [],
      change_points: changePoints
    };
  });
  const foundSignificantEventsIds = significantEvents.map(event => event.id);
  const notFoundSignificantEvents = queryLinks.filter(queryLink => !foundSignificantEventsIds.includes(queryLink.query.id)).map(queryLink => ({
    id: queryLink.query.id,
    title: queryLink.query.title,
    kql: queryLink.query.kql,
    feature: queryLink.query.feature,
    occurrences: [],
    change_points: {
      type: {
        stationary: {
          p_value: 0,
          change_point: 0
        }
      }
    }
  }));
  return [...significantEvents, ...notFoundSignificantEvents];
}