"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.stepConverter = exports.recalcColumnWidths = exports.isSetProcessor = exports.isManualIngestPipelineJsonProcessor = exports.isGrokProcessor = exports.isDissectProcessor = exports.isDateProcessor = exports.getValidSteps = exports.getStepPanelColour = exports.getFormStateFromWhereStep = exports.getFormStateFromActionStep = exports.getDefaultGrokProcessor = exports.getDefaultFormStateByType = exports.dataSourceConverter = exports.convertWhereBlockFormStateToConfiguration = exports.convertFormStateToProcessor = exports.SPECIALISED_TYPES = void 0;
var _streamsSchema = require("@kbn/streams-schema");
var _eui = require("@elastic/eui");
var _lodash = require("lodash");
var _grokUi = require("@kbn/grok-ui");
var _streamlang = require("@kbn/streamlang");
var _streamlang2 = require("@kbn/streamlang/types/streamlang");
var _config_driven = require("./steps/blocks/action/config_driven");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable @typescript-eslint/naming-convention */

/**
 * These are processor types with specialised UI. Other processor types are handled by a generic config-driven UI.
 */
const SPECIALISED_TYPES = exports.SPECIALISED_TYPES = ['date', 'dissect', 'grok', 'set'];
const PRIORITIZED_CONTENT_FIELDS = ['message', 'body.text', 'error.message', 'event.original', 'attributes.exception.message'];
const PRIORITIZED_DATE_FIELDS = ['timestamp', 'logtime', 'initial_date', 'date', 'event.time.received', 'event.ingested', 'custom.timestamp', 'attributes.custom.timestamp'];
const getDefaultTextField = (sampleDocs, prioritizedFields) => {
  // Count occurrences of well-known text fields in the sample documents
  const acceptableDefaultFields = sampleDocs.flatMap(doc => Object.keys(doc).filter(key => prioritizedFields.includes(key)));
  const acceptableFieldsOccurrences = (0, _lodash.countBy)(acceptableDefaultFields);

  // Sort by count descending first, then by order of field in prioritizedFields
  const sortedFields = (0, _lodash.orderBy)(Object.entries(acceptableFieldsOccurrences), [([_field, occurrencies]) => occurrencies,
  // Sort entries by occurrencies descending
  ([field]) => prioritizedFields.indexOf(field) // Sort entries by priority order in well-known fields
  ], ['desc', 'asc']);
  const mostCommonField = sortedFields[0];
  return mostCommonField ? mostCommonField[0] : '';
};
const defaultDateProcessorFormState = sampleDocs => ({
  action: 'date',
  from: getDefaultTextField(sampleDocs, PRIORITIZED_DATE_FIELDS),
  formats: [],
  to: '',
  output_format: '',
  ignore_failure: true,
  where: _streamlang.ALWAYS_CONDITION
});
const defaultDissectProcessorFormState = sampleDocs => ({
  action: 'dissect',
  from: getDefaultTextField(sampleDocs, PRIORITIZED_CONTENT_FIELDS),
  pattern: '',
  ignore_failure: true,
  ignore_missing: true,
  where: _streamlang.ALWAYS_CONDITION
});
const defaultGrokProcessorFormState = (sampleDocs, formStateDependencies) => ({
  action: 'grok',
  from: getDefaultTextField(sampleDocs, PRIORITIZED_CONTENT_FIELDS),
  patterns: [new _grokUi.DraftGrokExpression(formStateDependencies.grokCollection, '')],
  ignore_failure: true,
  ignore_missing: true,
  where: _streamlang.ALWAYS_CONDITION
});
const defaultManualIngestPipelineProcessorFormState = () => ({
  action: 'manual_ingest_pipeline',
  processors: [],
  ignore_failure: true,
  where: _streamlang.ALWAYS_CONDITION
});
const defaultSetProcessorFormState = () => ({
  action: 'set',
  to: '',
  value: '',
  ignore_failure: false,
  override: true,
  where: _streamlang.ALWAYS_CONDITION
});
const configDrivenDefaultFormStates = (0, _lodash.mapValues)(_config_driven.configDrivenProcessors, config => () => config.defaultFormState);
const defaultProcessorFormStateByType = {
  date: defaultDateProcessorFormState,
  dissect: defaultDissectProcessorFormState,
  grok: defaultGrokProcessorFormState,
  manual_ingest_pipeline: defaultManualIngestPipelineProcessorFormState,
  set: defaultSetProcessorFormState,
  ...configDrivenDefaultFormStates
};
const getDefaultFormStateByType = (type, sampleDocuments, formStateDependencies) => defaultProcessorFormStateByType[type](sampleDocuments, formStateDependencies);
exports.getDefaultFormStateByType = getDefaultFormStateByType;
const getFormStateFromActionStep = (sampleDocuments, formStateDependencies, step) => {
  if (!step) return defaultGrokProcessorFormState(sampleDocuments, formStateDependencies);
  if (step.action === 'grok') {
    const {
      customIdentifier,
      parentId,
      ...restStep
    } = step;
    const clone = structuredClone({
      ...(0, _lodash.omit)(restStep, 'patterns'),
      patterns: []
    });
    clone.patterns = step.patterns.map(pattern => new _grokUi.DraftGrokExpression(formStateDependencies.grokCollection, pattern));
    return clone;
  }
  if (step.action === 'dissect' || step.action === 'manual_ingest_pipeline' || step.action === 'date' || step.action === 'set') {
    const {
      customIdentifier,
      parentId,
      ...restStep
    } = step;
    return structuredClone({
      ...restStep
    });
  }
  if (step.action in _config_driven.configDrivenProcessors) {
    const {
      customIdentifier,
      parentId,
      ...restStep
    } = step;
    return _config_driven.configDrivenProcessors[step.action].convertProcessorToFormState(restStep);
  }
  throw new Error(`Form state for processor type "${step.action}" is not implemented.`);
};
exports.getFormStateFromActionStep = getFormStateFromActionStep;
const getFormStateFromWhereStep = step => {
  return structuredClone({
    ...step
  });
};
exports.getFormStateFromWhereStep = getFormStateFromWhereStep;
const convertWhereBlockFormStateToConfiguration = formState => {
  return {
    whereDefinition: {
      ...formState
    }
  };
};
exports.convertWhereBlockFormStateToConfiguration = convertWhereBlockFormStateToConfiguration;
const convertFormStateToProcessor = formState => {
  if ('action' in formState) {
    if (formState.action === 'grok') {
      const {
        patterns,
        from,
        ignore_failure,
        ignore_missing
      } = formState;
      return {
        processorDefinition: {
          action: 'grok',
          where: formState.where,
          patterns: patterns.map(pattern => pattern.getExpression().trim()).filter(pattern => !(0, _lodash.isEmpty)(pattern)),
          from,
          ignore_failure,
          ignore_missing
        },
        processorResources: {
          grokExpressions: patterns
        }
      };
    }
    if (formState.action === 'dissect') {
      const {
        from,
        pattern,
        append_separator,
        ignore_failure,
        ignore_missing
      } = formState;
      return {
        processorDefinition: {
          action: 'dissect',
          where: formState.where,
          from,
          pattern,
          append_separator: (0, _lodash.isEmpty)(append_separator) ? undefined : append_separator,
          ignore_failure,
          ignore_missing
        }
      };
    }
    if (formState.action === 'manual_ingest_pipeline') {
      const {
        processors,
        ignore_failure
      } = formState;
      return {
        processorDefinition: {
          action: 'manual_ingest_pipeline',
          where: formState.where,
          processors,
          ignore_failure
        }
      };
    }
    if (formState.action === 'date') {
      const {
        from,
        formats,
        ignore_failure,
        to,
        output_format
      } = formState;
      return {
        processorDefinition: {
          action: 'date',
          where: formState.where,
          from,
          formats,
          ignore_failure,
          to: (0, _lodash.isEmpty)(to) ? undefined : to,
          output_format: (0, _lodash.isEmpty)(output_format) ? undefined : output_format
        }
      };
    }
    if (formState.action === 'set') {
      const {
        to,
        value,
        copy_from,
        ignore_failure,
        override
      } = formState;
      const getValueOrCopyFrom = () => {
        if (typeof copy_from === 'string' && !(0, _lodash.isEmpty)(copy_from)) {
          return {
            copy_from
          };
        }
        if (typeof value === 'string' && !(0, _lodash.isEmpty)(value)) {
          return {
            value
          };
        }
        return {
          value: ''
        };
      };
      return {
        processorDefinition: {
          action: 'set',
          where: formState.where,
          to,
          ...getValueOrCopyFrom(),
          override,
          ignore_failure
        }
      };
    }
    if (_config_driven.configDrivenProcessors[formState.action]) {
      return {
        processorDefinition: _config_driven.configDrivenProcessors[formState.action].convertFormStateToConfig(formState)
      };
    }
  }
  throw new Error('Cannot convert form state to processing: unknown type.');
};
exports.convertFormStateToProcessor = convertFormStateToProcessor;
const createProcessorGuardByType = type => processor => processor.action === type;
const isDateProcessor = exports.isDateProcessor = createProcessorGuardByType('date');
const isDissectProcessor = exports.isDissectProcessor = createProcessorGuardByType('dissect');
const isManualIngestPipelineJsonProcessor = exports.isManualIngestPipelineJsonProcessor = createProcessorGuardByType('manual_ingest_pipeline');
const isGrokProcessor = exports.isGrokProcessor = createProcessorGuardByType('grok');
const isSetProcessor = exports.isSetProcessor = createProcessorGuardByType('set');
const createId = (0, _eui.htmlIdGenerator)();
const stepConverter = exports.stepConverter = {
  toUIDefinition: _streamlang.convertStepToUIDefinition
};
const dataSourceToUIDefinition = dataSource => ({
  id: createId(),
  ...dataSource
});
const dataSourceToUrlSchema = dataSourceWithUIAttributes => {
  const {
    id,
    ...dataSource
  } = dataSourceWithUIAttributes;
  return dataSource;
};
const dataSourceConverter = exports.dataSourceConverter = {
  toUIDefinition: dataSourceToUIDefinition,
  toUrlSchema: dataSourceToUrlSchema
};
const getDefaultGrokProcessor = ({
  sampleDocs
}) => ({
  action: 'grok',
  from: getDefaultTextField(sampleDocs, PRIORITIZED_CONTENT_FIELDS),
  patterns: [''],
  ignore_failure: true,
  ignore_missing: true,
  where: _streamlang.ALWAYS_CONDITION
});
exports.getDefaultGrokProcessor = getDefaultGrokProcessor;
const recalcColumnWidths = ({
  columnId,
  width,
  prevWidths,
  visibleColumns
}) => {
  const next = {
    ...prevWidths
  };
  if (width === undefined) {
    delete next[columnId];
  } else {
    next[columnId] = width;
  }
  const allExplicit = visibleColumns.every(c => next[c] !== undefined);
  if (allExplicit) {
    delete next[visibleColumns[visibleColumns.length - 1]];
  }
  return next;
};

// Get valid steps for simulation
// This will return valid action blocks, and valid where blocks, where
// where blocks are invalid all their children are also skipped.
exports.recalcColumnWidths = recalcColumnWidths;
const getValidSteps = steps => {
  const validSteps = [];

  // Helper to recursively skip invalid where blocks and their children
  function processStep(step) {
    if ((0, _streamlang2.isWhereBlock)(step)) {
      // If the where block is invalid, skip it and all its children
      if (!(0, _streamsSchema.isSchema)(_streamlang.conditionSchema, step.where)) {
        return false;
      }

      // Valid but has no children (compilation of this step would be pointless)
      const hasChildren = steps.some(s => s.parentId === step.customIdentifier);
      if (!hasChildren) {
        return false;
      }

      // Valid where block with children
      validSteps.push(step);
      return true;
    } else {
      // Action step: check validity
      if ((0, _streamsSchema.isSchema)(_streamlang.streamlangProcessorSchema, step)) {
        validSteps.push(step);
        return true;
      }
      return false;
    }
  }

  // We assume steps is a flat array, so we need to skip children of invalid where blocks
  const skipParentIds = new Set();
  for (const step of steps) {
    // If this step's parent is in skipParentIds, skip it (and its children)
    if (step.parentId && skipParentIds.has(step.parentId)) {
      skipParentIds.add(step.customIdentifier);
      continue;
    }
    const isValid = processStep(step);

    // If this is an invalid where block, add its id to skipParentIds
    if ((0, _streamlang2.isWhereBlock)(step) && !isValid) {
      skipParentIds.add(step.customIdentifier);
    }
  }
  return validSteps;
};
exports.getValidSteps = getValidSteps;
const getStepPanelColour = stepIndex => {
  const isEven = stepIndex % 2 === 0;
  return isEven ? 'subdued' : undefined;
};
exports.getStepPanelColour = getStepPanelColour;