"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSchemaFields = exports.getDefinitionFields = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _streamsSchema = require("@kbn/streams-schema");
var _lodash = require("lodash");
var _reactHooks = require("@kbn/react-hooks");
var _get_stream_type_from_definition = require("../../../../util/get_stream_type_from_definition");
var _use_streams_app_fetch = require("../../../../hooks/use_streams_app_fetch");
var _use_kibana = require("../../../../hooks/use_kibana");
var _utils = require("../utils");
var _errors = require("../../../../util/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useSchemaFields = ({
  definition,
  refreshDefinition
}) => {
  const {
    dependencies: {
      start: {
        streams: {
          streamsRepositoryClient
        },
        data: {
          dataViews
        }
      }
    },
    core: {
      notifications: {
        toasts
      }
    },
    services: {
      telemetryClient
    }
  } = (0, _use_kibana.useKibana)();
  const abortController = (0, _reactHooks.useAbortController)();
  const {
    value: unmappedFieldsValue,
    loading: isLoadingUnmappedFields,
    refresh: refreshUnmappedFields
  } = (0, _use_streams_app_fetch.useStreamsAppFetch)(({
    signal
  }) => {
    return streamsRepositoryClient.fetch('GET /internal/streams/{name}/schema/unmapped_fields', {
      signal,
      params: {
        path: {
          name: definition.stream.name
        }
      }
    });
  }, [definition.stream.name, streamsRepositoryClient]);
  const {
    value: dataViewFields,
    loading: isLoadingDataViewFields,
    refresh: refreshDataViewFields
  } = (0, _reactHooks.useAbortableAsync)(async ({
    signal
  }) => {
    return dataViews.getFieldsForWildcard({
      pattern: definition.stream.name,
      abortSignal: signal,
      forceRefresh: true
    });
  }, [dataViews, definition.stream.name]);
  const [fields, setFields] = (0, _react.useState)([]);
  const storedFields = (0, _react.useMemo)(() => {
    var _unmappedFieldsValue$;
    const definitionFields = getDefinitionFields(definition);
    const allManagedFieldsSet = new Set([...definitionFields.map(field => field.name)]);
    const unmanagedFields = dataViewFields ? dataViewFields.filter(field => !field.runtimeField && !field.metadata_field && !allManagedFieldsSet.has(field.name)).map(field => {
      var _field$esTypes;
      return {
        name: field.name,
        status: 'unmapped',
        esType: (_field$esTypes = field.esTypes) === null || _field$esTypes === void 0 ? void 0 : _field$esTypes[0],
        parent: definition.stream.name
      };
    }) : [];
    const allFoundFieldsSet = new Set([...definitionFields.map(field => field.name), ...unmanagedFields.map(field => field.name)]);
    const unmappedFields = (_unmappedFieldsValue$ = unmappedFieldsValue === null || unmappedFieldsValue === void 0 ? void 0 : unmappedFieldsValue.unmappedFields.filter(field => !allFoundFieldsSet.has(field)).map(field => ({
      name: field,
      parent: definition.stream.name,
      status: 'unmapped'
    }))) !== null && _unmappedFieldsValue$ !== void 0 ? _unmappedFieldsValue$ : [];
    return [...definitionFields, ...unmanagedFields, ...unmappedFields];
  }, [dataViewFields, definition, unmappedFieldsValue === null || unmappedFieldsValue === void 0 ? void 0 : unmappedFieldsValue.unmappedFields]);
  (0, _react.useEffect)(() => setFields(storedFields), [storedFields]);
  const refreshFields = (0, _react.useCallback)(() => {
    refreshDefinition();
    refreshUnmappedFields();
    refreshDataViewFields();
  }, [refreshDefinition, refreshUnmappedFields, refreshDataViewFields]);
  const addField = (0, _react.useCallback)(field => {
    setFields([...fields, field]);
  }, [fields]);
  const updateField = (0, _react.useCallback)(field => {
    setFields(prevFields => {
      const index = prevFields.findIndex(f => f.name === field.name);
      if (index === -1) {
        return prevFields;
      }
      const before = prevFields.slice(0, index);
      const after = prevFields.slice(index + 1);
      return [...before, field, ...after];
    });
  }, [setFields]);
  const pendingChangesCount = (0, _react.useMemo)(() => {
    const addedOrChanged = fields.filter(field => {
      const stored = storedFields.find(storedField => storedField.name === field.name);
      if (!stored) {
        return field.status !== 'unmapped';
      }
      return !(0, _lodash.isEqual)(field, stored);
    });
    return addedOrChanged.length;
  }, [fields, storedFields]);
  const discardChanges = (0, _react.useCallback)(() => {
    setFields(storedFields);
  }, [storedFields]);
  const submitChanges = (0, _react.useCallback)(async () => {
    try {
      const mappedFields = fields.filter(field => field.status === 'mapped').reduce((acc, field) => {
        acc[field.name] = (0, _utils.convertToFieldDefinitionConfig)(field);
        return acc;
      }, {});
      await streamsRepositoryClient.fetch(`PUT /api/streams/{name}/_ingest 2023-10-31`, {
        signal: abortController.signal,
        params: {
          path: {
            name: definition.stream.name
          },
          body: {
            ingest: {
              ...definition.stream.ingest,
              ...(_streamsSchema.Streams.WiredStream.GetResponse.is(definition) ? {
                wired: {
                  ...definition.stream.ingest.wired,
                  fields: mappedFields
                }
              } : {
                classic: {
                  ...definition.stream.ingest.classic,
                  field_overrides: mappedFields
                }
              })
            }
          }
        }
      });
      telemetryClient.trackSchemaUpdated({
        stream_type: (0, _get_stream_type_from_definition.getStreamTypeFromDefinition)(definition.stream)
      });
      toasts.addSuccess(_i18n.i18n.translate('xpack.streams.streamDetailSchemaEditorEditSuccessToast', {
        defaultMessage: 'Schema was successfully modified'
      }));
      refreshFields();
    } catch (error) {
      toasts.addError(new Error(error.body.message), {
        title: _i18n.i18n.translate('xpack.streams.streamDetailSchemaEditorEditErrorToast', {
          defaultMessage: 'Something went wrong when modifying the schema'
        }),
        toastMessage: (0, _errors.getFormattedError)(error).message,
        toastLifeTimeMs: 5000
      });
    }
  }, [fields, streamsRepositoryClient, abortController.signal, definition, telemetryClient, toasts, refreshFields]);
  return {
    fields,
    storedFields,
    isLoadingFields: isLoadingUnmappedFields || isLoadingDataViewFields,
    refreshFields,
    addField,
    updateField,
    pendingChangesCount,
    discardChanges,
    submitChanges
  };
};
exports.useSchemaFields = useSchemaFields;
const getDefinitionFields = definition => {
  let inheritedFields = [];
  if (_streamsSchema.Streams.WiredStream.GetResponse.is(definition)) {
    inheritedFields = Object.entries(definition.inherited_fields).map(([name, field]) => ({
      name,
      type: field.type,
      format: 'format' in field ? field.format : undefined,
      additionalParameters: (0, _streamsSchema.getAdvancedParameters)(name, field),
      parent: field.from,
      alias_for: field.alias_for,
      status: 'inherited'
    }));
  }
  const mappedFields = Object.entries(_streamsSchema.Streams.WiredStream.GetResponse.is(definition) ? definition.stream.ingest.wired.fields : definition.stream.ingest.classic.field_overrides || {}).map(([name, field]) => ({
    name,
    type: field.type,
    format: 'format' in field ? field.format : undefined,
    additionalParameters: (0, _streamsSchema.getAdvancedParameters)(name, field),
    parent: definition.stream.name,
    status: 'mapped'
  }));
  return [...mappedFields, ...inheritedFields];
};
exports.getDefinitionFields = getDefinitionFields;