"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateSignificantEvents = generateSignificantEvents;
var _inferenceCommon = require("@kbn/inference-common");
var _aiTools = require("@kbn/ai-tools");
var _streamlang = require("@kbn/streamlang");
var _inferencePromptUtils = require("@kbn/inference-prompt-utils");
var _esQuery = require("@kbn/es-query");
var _prompt = require("./prompt");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Generate significant event definitions, based on:
 * - the description of the feature (or stream if feature is undefined)
 * - dataset analysis
 * - for the given significant event types
 */
async function generateSignificantEvents({
  stream,
  feature,
  start,
  end,
  esClient,
  inferenceClient,
  logger
}) {
  const analysis = await (0, _aiTools.describeDataset)({
    start,
    end,
    esClient,
    index: stream.name,
    filter: feature !== null && feature !== void 0 && feature.filter ? (0, _streamlang.conditionToQueryDsl)(feature.filter) : undefined
  });
  const response = await (0, _inferencePromptUtils.executeAsReasoningAgent)({
    input: {
      name: (feature === null || feature === void 0 ? void 0 : feature.name) || stream.name,
      dataset_analysis: JSON.stringify((0, _aiTools.sortAndTruncateAnalyzedFields)(analysis, {
        dropEmpty: true
      })),
      description: (feature === null || feature === void 0 ? void 0 : feature.description) || stream.description
    },
    maxSteps: 4,
    prompt: _prompt.GenerateSignificantEventsPrompt,
    inferenceClient,
    toolCallbacks: {
      add_queries: async toolCall => {
        const queries = toolCall.function.arguments.queries;
        const queryValidationResults = queries.map(query => {
          let validation = {
            valid: true
          };
          try {
            (0, _esQuery.fromKueryExpression)(query.kql);
          } catch (error) {
            validation = {
              valid: false,
              error
            };
          }
          return {
            query,
            valid: validation.valid,
            status: validation.valid ? 'Added' : 'Failed to add',
            error: 'error' in validation ? validation.error.message : undefined
          };
        });
        return {
          response: {
            queries: queryValidationResults
          }
        };
      }
    }
  });
  const queries = response.input.flatMap(message => {
    if (message.role === _inferenceCommon.MessageRole.Tool) {
      return message.response.queries.flatMap(query => {
        if (query.valid) {
          return [query.query];
        }
        return [];
      });
    }
    return [];
  });
  return {
    queries
  };
}