"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useTestQuery = useTestQuery;
var _react = require("react");
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TEST_QUERY_INITIAL_RESPONSE = {
  result: null,
  error: null,
  warning: null,
  isLoading: false,
  preview: null
};

/**
 * Hook used to test the data fetching execution by returning a number of found documents
 * Or in error in case it's failing
 */
function useTestQuery(fetch) {
  const [testQueryResponse, setTestQueryResponse] = (0, _react.useState)(TEST_QUERY_INITIAL_RESPONSE);

  // Reset query response when criteria got changed
  (0, _react.useEffect)(() => {
    setTestQueryResponse(TEST_QUERY_INITIAL_RESPONSE);
  }, [fetch]);
  const onTestQuery = (0, _react.useCallback)(async () => {
    setTestQueryResponse({
      result: null,
      error: null,
      warning: null,
      isLoading: true,
      preview: null
    });
    try {
      const {
        testResults,
        isGrouped,
        isGroupedByRow,
        timeWindow,
        preview,
        warning
      } = await fetch();
      let trimmedPreview = null;
      if (preview) {
        trimmedPreview = {
          cols: preview.cols,
          rows: preview.rows.slice(0, 5)
        };
      }
      if (isGrouped) {
        const count = testResults.results.length;
        const result = isGroupedByRow ? _i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.testQueryGroupedByRowResponse', {
          defaultMessage: 'Query returned {rows} rows in the last {window}.',
          values: {
            rows: count,
            window: timeWindow
          }
        }) : _i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.testQueryGroupedResponse', {
          defaultMessage: 'Grouped query matched {groups} groups in the last {window}.',
          values: {
            groups: testResults.results.length,
            window: timeWindow
          }
        });
        setTestQueryResponse({
          result,
          error: null,
          warning: warning !== null && warning !== void 0 ? warning : null,
          isLoading: false,
          preview: trimmedPreview
        });
      } else {
        var _ungroupedQueryRespon;
        const ungroupedQueryResponse = testResults.results.length > 0 ? testResults.results[0] : {
          count: 0
        };
        setTestQueryResponse({
          result: _i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.numQueryMatchesText', {
            defaultMessage: 'Query matched {count} documents in the last {window}.',
            values: {
              count: (_ungroupedQueryRespon = ungroupedQueryResponse === null || ungroupedQueryResponse === void 0 ? void 0 : ungroupedQueryResponse.count) !== null && _ungroupedQueryRespon !== void 0 ? _ungroupedQueryRespon : 0,
              window: timeWindow
            }
          }),
          error: null,
          warning: null,
          isLoading: false,
          preview: trimmedPreview
        });
      }
    } catch (err) {
      var _err$body, _err$body$attributes, _err$body$attributes$, _err$body$attributes$2, _err$body2;
      const message = (err === null || err === void 0 ? void 0 : (_err$body = err.body) === null || _err$body === void 0 ? void 0 : (_err$body$attributes = _err$body.attributes) === null || _err$body$attributes === void 0 ? void 0 : (_err$body$attributes$ = _err$body$attributes.error) === null || _err$body$attributes$ === void 0 ? void 0 : (_err$body$attributes$2 = _err$body$attributes$.root_cause[0]) === null || _err$body$attributes$2 === void 0 ? void 0 : _err$body$attributes$2.reason) || (err === null || err === void 0 ? void 0 : (_err$body2 = err.body) === null || _err$body2 === void 0 ? void 0 : _err$body2.message);
      setTestQueryResponse({
        result: null,
        error: _i18n.i18n.translate('xpack.stackAlerts.esQuery.ui.queryError', {
          defaultMessage: 'Error testing query: {message}',
          values: {
            message: message ? `${err.message}: ${message}` : err.message
          }
        }),
        warning: null,
        isLoading: false,
        preview: null
      });
    }
  }, [fetch]);
  return {
    onTestQuery,
    testQueryResult: testQueryResponse.result,
    testQueryError: testQueryResponse.error,
    testQueryWarning: testQueryResponse.warning,
    testQueryLoading: testQueryResponse.isLoading,
    testQueryPreview: testQueryResponse.preview
  };
}