"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createObservableFromHttpResponse = createObservableFromHttpResponse;
var _eventsourceParser = require("eventsource-parser");
var _rxjs = require("rxjs");
var _sseUtils = require("@kbn/sse-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function createObservableFromHttpResponse(response) {
  const rawResponse = response.response;
  const body = rawResponse === null || rawResponse === void 0 ? void 0 : rawResponse.body;
  if (!body) {
    return (0, _rxjs.throwError)(() => {
      throw (0, _sseUtils.createSSEInternalError)(`No readable stream found in response`);
    });
  }
  return new _rxjs.Observable(subscriber => {
    const parser = (0, _eventsourceParser.createParser)({
      onEvent: event => {
        try {
          const data = JSON.parse(event.data);
          if (event.event === 'error') {
            const errorData = data;
            subscriber.error(new _sseUtils.ServerSentEventError(errorData.error.code, errorData.error.message, errorData.error.meta));
          } else {
            subscriber.next({
              type: event.event || 'event',
              ...data
            });
          }
        } catch (error) {
          subscriber.error((0, _sseUtils.createSSEInternalError)(`Failed to parse JSON`));
        }
      }
    });
    const readStream = async () => {
      const reader = body.getReader();
      const decoder = new TextDecoder();

      // Function to process each chunk
      const processChunk = ({
        done,
        value
      }) => {
        if (done) {
          return Promise.resolve();
        }
        parser.feed(decoder.decode(value, {
          stream: true
        }));
        return reader.read().then(processChunk);
      };

      // Start reading the stream
      return reader.read().then(processChunk);
    };
    readStream().then(() => {
      subscriber.complete();
    }).catch(error => {
      subscriber.error(error);
    });
  });
}