"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TimesliceMetricTransformGenerator = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _ = require(".");
var _constants = require("../../../common/constants");
var _slo_transform_template = require("../../assets/transform_templates/slo_transform_template");
var _errors = require("../../errors");
var _aggregations = require("../aggregations");
var _common = require("./common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const INVALID_EQUATION_REGEX = /[^A-Z|+|\-|\s|\d+|\.|\(|\)|\/|\*|>|<|=|\?|\:|&|\!|\|]+/g;
class TimesliceMetricTransformGenerator extends _.TransformGenerator {
  constructor(spaceId, dataViewService, isServerless) {
    super(spaceId, dataViewService, isServerless);
  }
  async getTransformParams(slo) {
    if (!_sloSchema.timesliceMetricIndicatorSchema.is(slo.indicator)) {
      throw new _errors.InvalidTransformError(`Cannot handle SLO of indicator type: ${slo.indicator.type}`);
    }
    return (0, _slo_transform_template.getSLOTransformTemplate)(this.buildTransformId(slo), this.buildDescription(slo), await this.buildSource(slo, slo.indicator), this.buildDestination(slo), this.buildCommonGroupBy(slo, slo.indicator.params.timestampField), this.buildAggregations(slo, slo.indicator), this.buildSettings(slo, slo.indicator.params.timestampField), slo);
  }
  buildTransformId(slo) {
    return (0, _constants.getSLOTransformId)(slo.id, slo.revision);
  }
  async buildSource(slo, indicator) {
    const dataView = await this.getIndicatorDataView(indicator.params.dataViewId);
    return {
      index: (0, _.parseIndex)(indicator.params.index),
      runtime_mappings: this.buildCommonRuntimeMappings(dataView),
      query: {
        bool: {
          filter: [(0, _common.getFilterRange)(slo, indicator.params.timestampField, this.isServerless), (0, _.getElasticsearchQueryOrThrow)(indicator.params.filter, dataView)]
        }
      }
    };
  }
  buildDestination(slo) {
    return {
      pipeline: (0, _constants.getSLOPipelineId)(slo.id, slo.revision),
      index: _constants.SLI_DESTINATION_INDEX_NAME
    };
  }
  buildAggregations(slo, indicator) {
    if (indicator.params.metric.equation.match(INVALID_EQUATION_REGEX)) {
      throw new Error(`Invalid equation: ${indicator.params.metric.equation}`);
    }
    if (!_sloSchema.timeslicesBudgetingMethodSchema.is(slo.budgetingMethod)) {
      throw new Error('The sli.metric.timeslice indicator MUST have a timeslice budgeting method.');
    }
    const getIndicatorAggregation = new _aggregations.GetTimesliceMetricIndicatorAggregation(indicator);
    const comparator = _sloSchema.timesliceMetricComparatorMapping[indicator.params.metric.comparator];
    return {
      ...getIndicatorAggregation.execute('_metric'),
      'slo.numerator': {
        bucket_script: {
          buckets_path: {
            value: '_metric>value'
          },
          script: {
            source: `params.value ${comparator} params.threshold ? 1 : 0`,
            params: {
              threshold: indicator.params.metric.threshold
            }
          }
        }
      },
      'slo.denominator': {
        bucket_script: {
          buckets_path: {},
          script: '1'
        }
      },
      'slo.isGoodSlice': {
        bucket_script: {
          buckets_path: {
            goodEvents: 'slo.numerator>value'
          },
          script: `params.goodEvents == 1 ? 1 : 0`
        }
      }
    };
  }
}
exports.TimesliceMetricTransformGenerator = TimesliceMetricTransformGenerator;