"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getElasticsearchQueryOrThrow = getElasticsearchQueryOrThrow;
exports.getFilterRange = getFilterRange;
exports.getTimesliceTargetComparator = getTimesliceTargetComparator;
exports.parseIndex = parseIndex;
exports.parseStringFilters = parseStringFilters;
var _esQuery = require("@kbn/es-query");
var _sloSchema = require("@kbn/slo-schema");
var _lodash = require("lodash");
var _get_delay_in_seconds_from_slo = require("../../domain/services/get_delay_in_seconds_from_slo");
var _errors = require("../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getElasticsearchQueryOrThrow(kuery = '', dataView) {
  try {
    if ((0, _lodash.isEmpty)(kuery)) {
      return {
        match_all: {}
      };
    }
    const kqlQuery = _sloSchema.kqlQuerySchema.is(kuery) ? kuery : kuery.kqlQuery;
    const filters = _sloSchema.kqlQuerySchema.is(kuery) ? [] : kuery.filters;
    return (0, _esQuery.buildEsQuery)(dataView, {
      query: kqlQuery,
      language: 'kuery'
    }, filters, {
      allowLeadingWildcards: true
    });
  } catch (err) {
    if (_sloSchema.kqlQuerySchema.is(kuery)) {
      throw new _errors.InvalidTransformError(`Invalid KQL: ${kuery}`);
    } else {
      throw new _errors.InvalidTransformError(`Invalid KQL: ${kuery.kqlQuery}`);
    }
  }
}
function parseStringFilters(filters, logger) {
  if (!filters) {
    return {};
  }
  try {
    return JSON.parse(filters);
  } catch (e) {
    logger.debug(`Failed to parse filters: ${e}`);
    return {};
  }
}
function parseIndex(index) {
  if (index.indexOf(',') === -1) {
    return index;
  }
  return index.split(',');
}
function getTimesliceTargetComparator(timesliceTarget) {
  return timesliceTarget === 0 ? '>' : '>=';
}

/**
 * Use the settings.preventInitialBackfill flag to determine the range filter for the rollup transform
 * preventInitialBackfill == true: we use the current time minus some buffer to account for the ingestion delay
 * preventInitialBackfill === false: we use the time window duration to get the data for the last N days
 */
function getFilterRange(slo, timestampField, isServerless) {
  if (slo.settings.preventInitialBackfill) {
    return {
      range: {
        [timestampField]: {
          gte: `now-${(0, _get_delay_in_seconds_from_slo.getDelayInSecondsFromSLO)(slo)}s/m`
        }
      }
    };
  }
  if (isServerless) {
    return {
      range: {
        [timestampField]: {
          gte: `now-7d`
        }
      }
    };
  }
  return {
    range: {
      [timestampField]: {
        gte: `now-${slo.timeWindow.duration.format()}/d`
      }
    }
  };
}