"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isPathValid = exports.hasSimpleExecutableName = exports.WILDCARD_WARNING = exports.TrustedDeviceConditionEntryField = exports.TRUSTED_DEVICE_OS_FIELD_AVAILABILITY = exports.OperatingSystem = exports.FILEPATH_WARNING = exports.EntryFieldType = exports.ConditionEntryField = void 0;
exports.isTrustedDeviceFieldAvailableForOs = isTrustedDeviceFieldAvailableForOs;
exports.validateWildcardInput = exports.validatePotentialWildcardInput = exports.validateHasWildcardWithWrongOperator = exports.validateFilePathInput = void 0;
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const WILDCARD_WARNING = exports.WILDCARD_WARNING = _i18n.i18n.translate('utils.wildcardWarning', {
  defaultMessage: `Using wildcards can impact Endpoint performance`
});
const FILEPATH_WARNING = exports.FILEPATH_WARNING = _i18n.i18n.translate('utils.filename.pathWarning', {
  defaultMessage: `Path may be formed incorrectly; verify value`
});
let ConditionEntryField = exports.ConditionEntryField = /*#__PURE__*/function (ConditionEntryField) {
  ConditionEntryField["HASH"] = "process.hash.*";
  ConditionEntryField["PATH"] = "process.executable.caseless";
  ConditionEntryField["SIGNER"] = "process.Ext.code_signature";
  ConditionEntryField["SIGNER_MAC"] = "process.code_signature";
  return ConditionEntryField;
}({});
let TrustedDeviceConditionEntryField = exports.TrustedDeviceConditionEntryField = /*#__PURE__*/function (TrustedDeviceConditionEntryField) {
  TrustedDeviceConditionEntryField["USERNAME"] = "user.name";
  TrustedDeviceConditionEntryField["HOST"] = "host.name";
  TrustedDeviceConditionEntryField["DEVICE_ID"] = "device.serial_number";
  TrustedDeviceConditionEntryField["MANUFACTURER"] = "device.vendor.name";
  TrustedDeviceConditionEntryField["PRODUCT_ID"] = "device.product.id";
  TrustedDeviceConditionEntryField["PRODUCT_NAME"] = "device.product.name";
  TrustedDeviceConditionEntryField["DEVICE_TYPE"] = "device.type";
  TrustedDeviceConditionEntryField["MANUFACTURER_ID"] = "device.vendor.id";
  return TrustedDeviceConditionEntryField;
}({});
let EntryFieldType = exports.EntryFieldType = /*#__PURE__*/function (EntryFieldType) {
  EntryFieldType["HASH"] = ".hash.";
  EntryFieldType["EXECUTABLE"] = ".executable.caseless";
  EntryFieldType["PATH"] = ".path";
  EntryFieldType["SIGNER"] = ".code_signature";
  return EntryFieldType;
}({});
let OperatingSystem = exports.OperatingSystem = /*#__PURE__*/function (OperatingSystem) {
  OperatingSystem["LINUX"] = "linux";
  OperatingSystem["MAC"] = "macos";
  OperatingSystem["WINDOWS"] = "windows";
  return OperatingSystem;
}({});
const TRUSTED_DEVICE_OS_FIELD_AVAILABILITY = exports.TRUSTED_DEVICE_OS_FIELD_AVAILABILITY = {
  /** Fields available for all supported operating systems */
  ALL_OS: [TrustedDeviceConditionEntryField.DEVICE_ID, TrustedDeviceConditionEntryField.DEVICE_TYPE, TrustedDeviceConditionEntryField.HOST, TrustedDeviceConditionEntryField.MANUFACTURER, TrustedDeviceConditionEntryField.MANUFACTURER_ID, TrustedDeviceConditionEntryField.PRODUCT_ID, TrustedDeviceConditionEntryField.PRODUCT_NAME],
  /** Fields available only for Windows OS exclusively */
  WINDOWS_ONLY: [TrustedDeviceConditionEntryField.USERNAME]
};
function isTrustedDeviceFieldAvailableForOs(field, osTypes) {
  const {
    WINDOWS_ONLY,
    ALL_OS
  } = TRUSTED_DEVICE_OS_FIELD_AVAILABILITY;
  if (WINDOWS_ONLY.includes(field)) {
    return osTypes.length === 1 && osTypes.includes(OperatingSystem.WINDOWS);
  }
  return ALL_OS.includes(field);
}
const validatePotentialWildcardInput = ({
  field = '',
  os,
  value = ''
}) => {
  const textInput = value.trim();
  if (field === 'file.path.text') {
    return validateFilePathInput({
      os,
      value: textInput
    });
  }
  return validateWildcardInput(textInput);
};
exports.validatePotentialWildcardInput = validatePotentialWildcardInput;
const validateFilePathInput = ({
  os,
  value
}) => {
  const isValidFilePath = isPathValid({
    os,
    field: 'file.path.text',
    type: 'wildcard',
    value
  });
  const hasSimpleFileName = hasSimpleExecutableName({
    os,
    type: 'wildcard',
    value
  });
  if (!value.length) {
    return FILEPATH_WARNING;
  }
  if (isValidFilePath) {
    if (hasSimpleFileName !== undefined && !hasSimpleFileName) {
      return WILDCARD_WARNING;
    }
  } else {
    return FILEPATH_WARNING;
  }
};
exports.validateFilePathInput = validateFilePathInput;
const validateWildcardInput = value => {
  const wildcardRegex = /[*?]/;
  if (Array.isArray(value)) {
    const doesAnyValueContainWildcardInput = value.some(v => wildcardRegex.test(v));
    if (doesAnyValueContainWildcardInput) {
      return WILDCARD_WARNING;
    }
  } else {
    if (wildcardRegex.test(value)) {
      return WILDCARD_WARNING;
    }
  }
};
exports.validateWildcardInput = validateWildcardInput;
const validateHasWildcardWithWrongOperator = ({
  operator,
  value
}) => {
  if (operator !== 'wildcard' && validateWildcardInput(value)) {
    return true;
  } else {
    return false;
  }
};
exports.validateHasWildcardWithWrongOperator = validateHasWildcardWithWrongOperator;
const hasSimpleExecutableName = ({
  os,
  type,
  value
}) => {
  const separator = os === OperatingSystem.WINDOWS ? '\\' : '/';
  const lastString = value.split(separator).pop();
  if (!lastString) {
    return;
  }
  if (type === 'wildcard') {
    return (lastString.split('*').length || lastString.split('?').length) === 1;
  }
  return true;
};
exports.hasSimpleExecutableName = hasSimpleExecutableName;
const isPathValid = ({
  os,
  field,
  type,
  value
}) => {
  const pathFields = ['process.executable.caseless', 'file.path', 'file.path.text'];
  if (pathFields.includes(field)) {
    if (type === 'wildcard') {
      return os === OperatingSystem.WINDOWS ? isWindowsWildcardPathValid(value) : isLinuxMacWildcardPathValid(value);
    }
    return doesPathMatchRegex({
      value,
      os
    });
  }
  return true;
};
exports.isPathValid = isPathValid;
const doesPathMatchRegex = ({
  os,
  value
}) => {
  if (os === OperatingSystem.WINDOWS) {
    const filePathRegex = /^[a-z]:(?:|\\\\[^<>:"'/\\|?*]+\\[^<>:"'/\\|?*]+|%\w+%|)[\\](?:[^<>:"'/\\|?*]+[\\/])*([^<>:"'/\\|?*])+$/i;
    return filePathRegex.test(value);
  }
  return /^(\/|(\/[\w\-]+)+|\/[\w\-]+\.[\w]+|(\/[\w-]+)+\/[\w\-]+\.[\w]+)$/i.test(value);
};
const isWindowsWildcardPathValid = path => {
  const firstCharacter = path[0];
  const lastCharacter = path.slice(-1);
  const trimmedValue = path.trim();
  const hasSlash = /\//.test(trimmedValue);
  if (path.length === 0) {
    return false;
  } else if (hasSlash || trimmedValue.length !== path.length || firstCharacter === '^' || lastCharacter === '\\' || !hasWildcardInPath({
    path,
    isWindowsPath: true
  })) {
    return false;
  } else {
    return true;
  }
};
const isLinuxMacWildcardPathValid = path => {
  const firstCharacter = path[0];
  const lastCharacter = path.slice(-1);
  const trimmedValue = path.trim();
  if (path.length === 0) {
    return false;
  } else if (trimmedValue.length !== path.length || firstCharacter !== '/' || lastCharacter === '/' || path.length > 1024 === true || path.includes('//') === true || !hasWildcardInPath({
    path,
    isWindowsPath: false
  })) {
    return false;
  } else {
    return true;
  }
};
const hasWildcardInPath = ({
  path,
  isWindowsPath
}) => {
  for (const pathComponent of path.split(isWindowsPath ? '\\' : '/')) {
    if (/[\*|\?]+/.test(pathComponent) === true) {
      return true;
    }
  }
  return false;
};