"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createTelemetryCustomResponseActionRulesTaskConfig = createTelemetryCustomResponseActionRulesTaskConfig;
var _lodash = require("lodash");
var _helpers = require("../helpers");
var _types = require("../types");
var _configuration = require("../configuration");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createTelemetryCustomResponseActionRulesTaskConfig(maxTelemetryBatch) {
  const taskName = 'Security Solution Response Actions Rules Telemetry';
  const taskType = 'security:telemetry-response-actions-rules';
  return {
    type: taskType,
    title: taskName,
    interval: '24h',
    timeout: '10m',
    version: '1.0.0',
    runTask: async (taskId, logger, receiver, sender, taskMetricsService, taskExecutionPeriod) => {
      const mdc = {
        task_id: taskId,
        task_execution_period: taskExecutionPeriod
      };
      const log = (0, _helpers.newTelemetryLogger)(logger.get('response_actions_rules'), mdc);
      const usageCollector = sender.getTelemetryUsageCluster();
      const usageLabelEndpointPrefix = ['security_telemetry', 'endpoint-response-actions-rules'];
      const usageLabelOsqueryPrefix = ['security_telemetry', 'osquery-response-actions-rules'];
      const trace = taskMetricsService.start(taskType);
      log.debug('Running response actions rules telemetry task');
      try {
        var _taskExecutionPeriod$;
        const [clusterInfoPromise, licenseInfoPromise] = await Promise.allSettled([receiver.fetchClusterInfo(), receiver.fetchLicenseInfo()]);
        const clusterInfo = (0, _helpers.safeValue)(clusterInfoPromise);
        const licenseInfo = (0, _helpers.safeValue)(licenseInfoPromise);
        const {
          body: {
            aggregations
          }
        } = await receiver.fetchResponseActionsRules((_taskExecutionPeriod$ = taskExecutionPeriod.last) !== null && _taskExecutionPeriod$ !== void 0 ? _taskExecutionPeriod$ : 'now-24h', taskExecutionPeriod.current);
        if (!aggregations || !aggregations.actionTypes) {
          log.debug('no custom response action rules found');
          await taskMetricsService.end(trace);
          return 0;
        }
        const responseActionRules = aggregations.actionTypes.buckets.reduce((acc, agg) => {
          if (agg.key === '.endpoint') {
            acc.endpoint = agg.doc_count;
          } else if (agg.key === '.osquery') {
            acc.osquery = agg.doc_count;
          }
          return acc;
        }, {
          endpoint: 0,
          osquery: 0
        });
        const shouldNotProcessTelemetry = responseActionRules.endpoint === 0 || responseActionRules.osquery === 0;
        if (shouldNotProcessTelemetry) {
          log.debug('no new custom response action rules found');
          await taskMetricsService.end(trace);
          return 0;
        }
        const responseActionsRulesTelemetryData = (0, _helpers.responseActionsCustomRuleTelemetryData)(responseActionRules, clusterInfo, licenseInfo);
        log.debug('Custom response actions rules data', {
          data: JSON.stringify(responseActionsRulesTelemetryData)
        });
        usageCollector === null || usageCollector === void 0 ? void 0 : usageCollector.incrementCounter({
          counterName: (0, _helpers.createUsageCounterLabel)(usageLabelEndpointPrefix),
          counterType: 'response_actions_endpoint_rules_count',
          incrementBy: responseActionsRulesTelemetryData.response_actions_rules.endpoint
        });
        usageCollector === null || usageCollector === void 0 ? void 0 : usageCollector.incrementCounter({
          counterName: (0, _helpers.createUsageCounterLabel)(usageLabelOsqueryPrefix),
          counterType: 'response_actions_osquery_rules_count',
          incrementBy: responseActionsRulesTelemetryData.response_actions_rules.osquery
        });
        const documents = (0, _lodash.cloneDeep)(Object.values(responseActionsRulesTelemetryData));
        if (_configuration.telemetryConfiguration.use_async_sender) {
          sender.sendAsync(_types.TelemetryChannel.LISTS, documents);
        } else {
          const batches = (0, _helpers.batchTelemetryRecords)(documents, maxTelemetryBatch);
          for (const batch of batches) {
            await sender.sendOnDemand(_types.TelemetryChannel.LISTS, batch);
          }
        }
        await taskMetricsService.end(trace);
        const totalCount = Object.values(responseActionsRulesTelemetryData.response_actions_rules).reduce((acc, count) => acc + count, 0);
        log.debug('Response actions rules telemetry task executed', {
          totalCount
        });
        return totalCount;
      } catch (error) {
        log.warn('Error running custom response actions rule task', (0, _helpers.withErrorMessage)(error));
        await taskMetricsService.end(trace, error);
        return 0;
      }
    }
  };
}