"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RuleMigrationTaskRunner = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _agent = require("./agent");
var _retrievers = require("./retrievers");
var _esql_knowledge_base = require("../../common/task/util/esql_knowledge_base");
var _nullify_missing_properties = require("../../common/task/util/nullify_missing_properties");
var _siem_migrations_task_runner = require("../../common/task/siem_migrations_task_runner");
var _rule_migrations_telemetry_client = require("./rule_migrations_telemetry_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RuleMigrationTaskRunner extends _siem_migrations_task_runner.SiemMigrationTaskRunner {
  constructor(migrationId, request, startedBy, abortController, data, logger, dependencies) {
    super(migrationId, request, startedBy, abortController, data, logger, dependencies);
    (0, _defineProperty2.default)(this, "retriever", void 0);
    (0, _defineProperty2.default)(this, "taskConcurrency", 10);
    this.migrationId = migrationId;
    this.request = request;
    this.startedBy = startedBy;
    this.abortController = abortController;
    this.data = data;
    this.logger = logger;
    this.dependencies = dependencies;
    this.retriever = new _retrievers.RuleMigrationsRetriever(this.migrationId, {
      data: this.data,
      rules: this.dependencies.rulesClient,
      savedObjects: this.dependencies.savedObjectsClient
    });
  }

  /** Retrieves the connector and creates the migration agent */
  async setup(connectorId) {
    const {
      inferenceService
    } = this.dependencies;
    const model = await this.actionsClientChat.createModel({
      connectorId,
      migrationId: this.migrationId,
      migrationType: 'rules',
      abortController: this.abortController
    });
    const modelName = this.actionsClientChat.getModelName(model);
    const telemetryClient = new _rule_migrations_telemetry_client.RuleMigrationTelemetryClient(this.dependencies.telemetry, this.logger, this.migrationId, modelName);
    const esqlKnowledgeBase = new _esql_knowledge_base.EsqlKnowledgeBase(connectorId, this.migrationId, inferenceService.getClient({
      request: this.request
    }), this.logger);
    const agent = (0, _agent.getRuleMigrationAgent)({
      esqlKnowledgeBase,
      model,
      ruleMigrationsRetriever: this.retriever,
      logger: this.logger,
      telemetryClient
    });
    this.telemetry = telemetryClient;
    this.task = (input, config) => agent.invoke(input, config);
  }

  /** Initializes the retriever populating ELSER indices. It may take a few minutes */
  async initialize() {
    await this.retriever.initialize();
  }
  async prepareTaskInput(migrationRule) {
    const resources = await this.retriever.resources.getResources(migrationRule.original_rule);
    return {
      id: migrationRule.id,
      original_rule: migrationRule.original_rule,
      resources
    };
  }

  /**
   *
   * Processes the output of the task for indexing.
   *
   * we use the nullify properties which are available in source (i.e. indexed document)
   * but missing in migration output.
   *
   * If a property is missing in output it means that it needs to be empty in the ES document as well
   * and only way to do that is to set it to null explicitly.
   *
   * */
  processTaskOutput(migrationRule, migrationOutput) {
    return {
      ...migrationRule,
      elastic_rule: (0, _nullify_missing_properties.nullifyMissingProperties)({
        source: migrationRule.elastic_rule,
        target: migrationOutput.elastic_rule
      }),
      translation_result: migrationOutput.translation_result,
      comments: migrationOutput.comments
    };
  }
}
exports.RuleMigrationTaskRunner = RuleMigrationTaskRunner;