"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RuleMigrationSpaceIndexMigrator = void 0;
var _assert = _interopRequireDefault(require("assert"));
var _constants = require("../../common/data/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RuleMigrationSpaceIndexMigrator {
  constructor(spaceId, esClient, logger, ruleMigrationIndexAdapters) {
    this.spaceId = spaceId;
    this.esClient = esClient;
    this.logger = logger;
    this.ruleMigrationIndexAdapters = ruleMigrationIndexAdapters;
  }

  /**
   * Runs the migrators for the rule migration index in the specified space.
   * It migrates existing rules to create migration documents and populates migration names.
   *
   * If any errors occur they are logged but should not prevent the server from starting.
   */
  async run() {
    this.logger.debug(`Starting migrators for space ${this.spaceId}`);
    try {
      await this.migrateRuleMigrationIndex();
    } catch (error) {
      this.logger.error(`Error migrating rule migration index for space ${this.spaceId}: ${error.message}`);
    }
    try {
      await this.populateMigrationNames();
    } catch (error) {
      this.logger.error(`Error populating migration names for space ${this.spaceId}: ${error.message}`);
    }
    this.logger.debug(`Finished migrators for space ${this.spaceId}`);
  }

  /**
   * Migrates the rule migration index by creating migration documents for existing rules
   * that do not have corresponding migration documents in the migrations index.
   */
  async migrateRuleMigrationIndex() {
    const installedIndexName = await this.ruleMigrationIndexAdapters.migrations.getInstalledIndexName(this.spaceId);
    if (!installedIndexName) {
      await this.ruleMigrationIndexAdapters.migrations.createIndex(this.spaceId);
    }
    const migrationsFromRulesIndex = await this.getMigrationFromRulesIndex();
    const migrationsFromMigrationsIndex = await this.getMigrationIdsFromMigrationsIndex();
    const migrationsToIndex = migrationsFromRulesIndex.filter(migration => !migrationsFromMigrationsIndex.some(id => id === migration.id));
    if (migrationsToIndex.length > 0) {
      await this.createMigrationDocs(migrationsToIndex.map(migration => {
        var _migration$created_by, _migration$created_at;
        return {
          ...migration,
          created_by: (_migration$created_by = migration.created_by) !== null && _migration$created_by !== void 0 ? _migration$created_by : '',
          created_at: (_migration$created_at = migration.created_at) !== null && _migration$created_at !== void 0 ? _migration$created_at : new Date().toISOString()
        };
      }));
      this.logger.debug(`Created ${migrationsToIndex.length} migration documents missing.`);
    }
  }

  /**
   * Populates migration documents that do not have a name field with generated names.
   * The names are generated based on the migration creation order, like the existing migrations are named in the runtime.
   */
  async populateMigrationNames() {
    const migrationIdsWithoutName = await this.getMigrationIdsWithoutName();
    if (migrationIdsWithoutName.length > 0) {
      const namesMap = await this.getMigrationsNamesMap();
      const migrationsToUpdate = migrationIdsWithoutName.map(id => {
        var _namesMap$get;
        const name = (_namesMap$get = namesMap.get(id)) !== null && _namesMap$get !== void 0 ? _namesMap$get : `SIEM Migration ${id}`; // Fallback name using the ID (should never happen, but just in case)
        return {
          id,
          name
        };
      });
      await this.updateMigrationDocs(migrationsToUpdate);
      this.logger.debug(`Updated ${migrationsToUpdate.length} migrations with generated name.`);
    }
  }

  /**
   * Creates migration documents in the migrations index.
   */
  async createMigrationDocs(docs) {
    const _index = this.ruleMigrationIndexAdapters.migrations.getIndexName(this.spaceId);
    const operations = docs.flatMap(({
      id: _id,
      ...doc
    }) => [{
      create: {
        _id,
        _index
      }
    }, {
      ...doc
    }]);
    return this.esClient.bulk({
      refresh: 'wait_for',
      operations
    });
  }

  /**
   * Updates migration documents in the migrations index.
   */
  async updateMigrationDocs(docs) {
    const _index = this.ruleMigrationIndexAdapters.migrations.getIndexName(this.spaceId);
    const operations = docs.flatMap(({
      id: _id,
      ...doc
    }) => [{
      update: {
        _id,
        _index
      }
    }, {
      doc
    }]);
    return this.esClient.bulk({
      refresh: 'wait_for',
      operations
    });
  }

  /**
   * Retrieves existing migrations from the rules index.
   * It aggregates by migration_id and returns the earliest created_at and created_by for each migration.
   * Results are ordered by `@timestamp` in ascending order. This is important for name generation
   */
  async getMigrationFromRulesIndex() {
    var _result$aggregations, _ref;
    const index = this.ruleMigrationIndexAdapters.rules.getIndexName(this.spaceId);
    const aggregations = {
      migrationIds: {
        terms: {
          field: 'migration_id',
          order: {
            createdAt: 'asc'
          },
          size: _constants.MAX_ES_SEARCH_SIZE
        },
        aggregations: {
          createdAt: {
            min: {
              field: '@timestamp'
            }
          },
          createdBy: {
            terms: {
              field: 'created_by'
            }
          }
        }
      }
    };
    const result = await this.esClient.search({
      index,
      aggregations,
      _source: false
    });
    const migrationsAgg = (_result$aggregations = result.aggregations) === null || _result$aggregations === void 0 ? void 0 : _result$aggregations.migrationIds;
    const buckets = (_ref = migrationsAgg === null || migrationsAgg === void 0 ? void 0 : migrationsAgg.buckets) !== null && _ref !== void 0 ? _ref : [];
    return buckets.map(bucket => {
      var _bucket$createdAt;
      return {
        id: `${bucket.key}`,
        created_at: (_bucket$createdAt = bucket.createdAt) === null || _bucket$createdAt === void 0 ? void 0 : _bucket$createdAt.value_as_string,
        created_by: bucket.createdBy.buckets[0].key
      };
    });
  }

  /**
   * Retrieves existing migrations from the migrations index.
   * It returns the IDs of all migration documents.
   */
  async getMigrationIdsFromMigrationsIndex() {
    const index = this.ruleMigrationIndexAdapters.migrations.getIndexName(this.spaceId);
    const result = await this.esClient.search({
      index,
      size: _constants.MAX_ES_SEARCH_SIZE,
      query: {
        match_all: {}
      },
      _source: false
    });
    return result.hits.hits.map(({
      _id
    }) => {
      (0, _assert.default)(_id, 'document should have _id');
      return _id;
    });
  }

  /**
   * Retrieves migration IDs from the migrations index that do not have a name field.
   */
  async getMigrationIdsWithoutName() {
    const index = this.ruleMigrationIndexAdapters.migrations.getIndexName(this.spaceId);
    const result = await this.esClient.search({
      index,
      query: {
        bool: {
          must_not: {
            exists: {
              field: 'name'
            }
          }
        }
      },
      size: _constants.MAX_ES_SEARCH_SIZE,
      _source: false
    });
    return result.hits.hits.map(({
      _id
    }) => {
      (0, _assert.default)(_id, 'document should have _id');
      return _id;
    });
  }

  /**
   * Creates the mapping of names by id for all the migrations from the rules index.
   * The names are generated based in creation order.
   */
  async getMigrationsNamesMap() {
    // Cache the names map to avoid repeat the aggregation query
    const migrations = await this.getMigrationFromRulesIndex();
    const namesMap = migrations.reduce((acc, migration, i) => {
      acc.set(migration.id, `SIEM rules migration #${i + 1}`);
      return acc;
    }, new Map());
    return namesMap;
  }
}
exports.RuleMigrationSpaceIndexMigrator = RuleMigrationSpaceIndexMigrator;