"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSiemDashboardMigrationsCreateDashboardsRoute = void 0;
var _zodHelpers = require("@kbn/zod-helpers");
var _dashboard_migration = require("../../../../../../common/siem_migrations/model/api/dashboards/dashboard_migration.gen");
var _constants = require("../../../../../../common/siem_migrations/dashboards/constants");
var _authz = require("../../../common/api/util/authz");
var _with_license = require("../../../common/api/util/with_license");
var _resources = require("../../../../../../common/siem_migrations/dashboards/resources");
var _audit = require("../../../common/api/util/audit");
var _with_existing_migration_id = require("../../../common/api/util/with_existing_migration_id");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerSiemDashboardMigrationsCreateDashboardsRoute = (router, logger) => {
  router.versioned.post({
    path: _constants.SIEM_DASHBOARD_MIGRATION_DASHBOARDS_PATH,
    access: 'internal',
    security: {
      authz: _authz.authz
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: (0, _zodHelpers.buildRouteValidationWithZod)(_dashboard_migration.CreateDashboardMigrationDashboardsRequestParams),
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_dashboard_migration.CreateDashboardMigrationDashboardsRequestBody)
      }
    }
  }, (0, _with_license.withLicense)((0, _with_existing_migration_id.withExistingMigration)(async (context, req, res) => {
    const {
      migration_id: migrationId
    } = req.params;
    const originalDashboardsExport = req.body;
    const originalDashboardsCount = originalDashboardsExport.length;
    if (originalDashboardsCount === 0) {
      return res.badRequest({
        body: `No dashboards provided for migration ID ${migrationId}. Please provide at least one dashboard.`
      });
    }
    const siemMigrationAuditLogger = new _audit.SiemMigrationAuditLogger(context.securitySolution, 'dashboards');
    try {
      const ctx = await context.resolve(['securitySolution']);
      const dashboardMigrationsClient = ctx.securitySolution.siemMigrations.getDashboardsClient();

      // Convert the original splunk dashboards format to the migration dashboard item document format
      const items = originalDashboardsExport.map(({
        result: {
          ...originalDashboard
        }
      }) => {
        var _originalDashboard$la, _originalDashboard$de;
        return {
          migration_id: migrationId,
          original_dashboard: {
            id: originalDashboard.id,
            title: (_originalDashboard$la = originalDashboard.label) !== null && _originalDashboard$la !== void 0 ? _originalDashboard$la : originalDashboard.title,
            description: (_originalDashboard$de = originalDashboard.description) !== null && _originalDashboard$de !== void 0 ? _originalDashboard$de : '',
            data: originalDashboard['eai:data'],
            format: 'xml',
            vendor: 'splunk',
            last_updated: originalDashboard.updated,
            splunk_properties: {
              app: originalDashboard['eai:acl.app'],
              owner: originalDashboard['eai:acl.owner'],
              sharing: originalDashboard['eai:acl.sharing']
            }
          }
        };
      });
      const resourceIdentifier = new _resources.DashboardResourceIdentifier(items[0].original_dashboard.vendor);
      const [, extractedResources] = await Promise.all([siemMigrationAuditLogger.logAddDashboards({
        migrationId,
        count: originalDashboardsCount
      }), resourceIdentifier.fromOriginals(items.map(dash => dash.original_dashboard))]);
      const resources = extractedResources.map(resource => ({
        ...resource,
        migration_id: migrationId
      }));
      await Promise.all([dashboardMigrationsClient.data.items.create(items), dashboardMigrationsClient.data.resources.create(resources)]);
      return res.ok();
    } catch (error) {
      logger.error(`Error creating dashboards for migration ID ${migrationId}: ${error}`);
      return res.customError({
        statusCode: 500,
        body: `Error creating dashboards for migration ID ${migrationId}: ${error.message}`
      });
    }
  })));
};
exports.registerSiemDashboardMigrationsCreateDashboardsRoute = registerSiemDashboardMigrationsCreateDashboardsRoute;