"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.withRiskEnginePrivilegeCheck = exports.getRunRiskEnginePrivileges = exports.getEnableRiskEnginePrivileges = exports._getMissingPrivilegesMessage = void 0;
var _routes = require("@kbn/lists-plugin/server/routes");
var _i18n = require("@kbn/i18n");
var _risk_engine = require("../../../../common/entity_analytics/risk_engine");
var _check_and_format_privileges = require("../utils/check_and_format_privileges");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getRunRiskEnginePrivileges = async (request, security) => {
  return (0, _check_and_format_privileges.checkAndFormatPrivileges)({
    request,
    security,
    privilegesToCheck: {
      elasticsearch: {
        cluster: _risk_engine.TO_RUN_RISK_ENGINE_REQUIRED_ES_CLUSTER_PRIVILEGES,
        index: {}
      }
    }
  });
};
exports.getRunRiskEnginePrivileges = getRunRiskEnginePrivileges;
const getEnableRiskEnginePrivileges = async (request, security) => {
  return (0, _check_and_format_privileges.checkAndFormatPrivileges)({
    request,
    security,
    privilegesToCheck: {
      elasticsearch: {
        cluster: _risk_engine.TO_ENABLE_RISK_ENGINE_REQUIRED_ES_CLUSTER_PRIVILEGES,
        index: _risk_engine.RISK_ENGINE_REQUIRED_ES_INDEX_PRIVILEGES
      }
    }
  });
};
exports.getEnableRiskEnginePrivileges = getEnableRiskEnginePrivileges;
const _getMissingPrivilegesMessage = riskEnginePrivileges => {
  const {
    indexPrivileges,
    clusterPrivileges
  } = (0, _risk_engine.getMissingRiskEnginePrivileges)(riskEnginePrivileges.privileges);
  const indexPrivilegesMessage = indexPrivileges.map(([indexName, privileges]) => _i18n.i18n.translate('xpack.securitySolution.entityAnalytics.riskEngine.missingIndexPrivilege', {
    defaultMessage: 'Missing index privileges for index "{indexName}": {privileges}.',
    values: {
      indexName,
      privileges: privileges.join(', ')
    }
  })).join('\n');
  const clusterRunPrivilegesMessage = !clusterPrivileges.run.length ? '' : _i18n.i18n.translate('xpack.securitySolution.entityAnalytics.riskEngine.missingClusterRunPrivilege', {
    defaultMessage: 'Missing cluster privileges to run the risk engine: {privileges}.',
    values: {
      privileges: clusterPrivileges.run.join(', ')
    }
  });
  const clusterEnablePrivilegesMessage = !clusterPrivileges.enable.length ? '' : _i18n.i18n.translate('xpack.securitySolution.entityAnalytics.riskEngine.missingClusterEnablePrivilege', {
    defaultMessage: 'Missing cluster privileges to enable the risk engine: {privileges}.',
    values: {
      privileges: clusterPrivileges.enable.join(', ')
    }
  });
  const unauthorizedMessage = _i18n.i18n.translate('xpack.securitySolution.entityAnalytics.riskEngine.unauthorized', {
    defaultMessage: 'User is missing risk engine privileges.'
  });
  return `${unauthorizedMessage} ${indexPrivilegesMessage} ${clusterRunPrivilegesMessage} ${clusterEnablePrivilegesMessage}`;
};

/**
 * This function is used to check if the user has the required privileges to access the risk engine.
 * It is used to wrap a risk engine route handler which requires full access to the risk engine.
 * @param getStartServices - Kibana's start services accessor
 * @param handler - The route handler to wrap
 **/
exports._getMissingPrivilegesMessage = _getMissingPrivilegesMessage;
const withRiskEnginePrivilegeCheck = (privilegeTypeOrServices, handlerOrServices, optionalHandler) => {
  // Determine if privilegeType is provided or if it's the default case
  let privilegeType = 'enable';
  let getStartServices;
  let handler;
  if (typeof privilegeTypeOrServices === 'string') {
    // First parameter is the privilegeType
    privilegeType = privilegeTypeOrServices;
    getStartServices = handlerOrServices;
    if (optionalHandler === undefined) {
      throw new Error('Handler is required when using privilege type parameter');
    }
    handler = optionalHandler;
  } else {
    // First parameter is getStartServices, privilegeType is default 'enable'
    getStartServices = privilegeTypeOrServices;
    handler = handlerOrServices;
  }
  return async (context, request, response) => {
    const [_, {
      security
    }] = await getStartServices();
    const privilegeCheckFn = privilegeType === 'run' ? getRunRiskEnginePrivileges : getEnableRiskEnginePrivileges;
    const privileges = await privilegeCheckFn(request, security);
    if (!privileges.has_all_required) {
      const siemResponse = (0, _routes.buildSiemResponse)(response);
      return siemResponse.error({
        statusCode: 403,
        body: _getMissingPrivilegesMessage(privileges)
      });
    }
    return handler(context, request, response);
  };
};
exports.withRiskEnginePrivilegeCheck = withRiskEnginePrivilegeCheck;