"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getApiKeyManager = void 0;
var _auth = require("@kbn/entityManager-plugin/server/lib/auth");
var _saved_objects = require("@kbn/entityManager-plugin/server/saved_objects");
var _fake_kibana_request = require("@kbn/security-plugin/server/authentication/api_keys/fake_kibana_request");
var _saved_object = require("@kbn/entityManager-plugin/server/lib/auth/api_key/saved_object");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getApiKeyManager = ({
  core,
  logger,
  security,
  encryptedSavedObjects,
  request,
  namespace
}) => ({
  generate: async () => {
    if (!encryptedSavedObjects) {
      throw new Error('Unable to create API key. Ensure encrypted Saved Object client is enabled in this environment.');
    } else if (!request) {
      throw new Error('Unable to create API key due to invalid request');
    } else {
      const apiKey = await (0, _auth.generateEntityDiscoveryAPIKey)({
        core,
        config: {},
        logger,
        security,
        encryptedSavedObjects
      }, request);
      const soClient = core.savedObjects.getScopedClient(request, {
        excludedExtensions: [_coreSavedObjectsServer.SECURITY_EXTENSION_ID],
        includedHiddenTypes: [_saved_objects.EntityDiscoveryApiKeyType.name]
      });
      await soClient.create(_saved_objects.EntityDiscoveryApiKeyType.name, apiKey, {
        id: (0, _saved_object.getSpaceAwareEntityDiscoverySavedObjectId)(namespace),
        overwrite: true,
        managed: true
      });
    }
  },
  getApiKey: async () => {
    if (!encryptedSavedObjects) {
      throw Error('Unable to retrieve API key. Ensure encrypted Saved Object client is enabled in this environment.');
    }
    try {
      const encryptedSavedObjectsClient = encryptedSavedObjects.getClient({
        includedHiddenTypes: [_saved_objects.EntityDiscoveryApiKeyType.name]
      });
      return (await encryptedSavedObjectsClient.getDecryptedAsInternalUser(_saved_objects.EntityDiscoveryApiKeyType.name, (0, _saved_object.getSpaceAwareEntityDiscoverySavedObjectId)(namespace), {
        namespace
      })).attributes;
    } catch (err) {
      if (_coreSavedObjectsServer.SavedObjectsErrorHelpers.isNotFoundError(err)) {
        return undefined;
      }
      throw err;
    }
  },
  getRequestFromApiKey: async apiKey => {
    return (0, _fake_kibana_request.getFakeKibanaRequest)({
      id: apiKey.id,
      api_key: apiKey.apiKey
    });
  },
  getClientFromApiKey: async apiKey => {
    const fakeRequest = (0, _fake_kibana_request.getFakeKibanaRequest)({
      id: apiKey.id,
      api_key: apiKey.apiKey
    });
    const clusterClient = core.elasticsearch.client.asScoped(fakeRequest);
    const soClient = core.savedObjects.getScopedClient(fakeRequest, {
      includedHiddenTypes: [_saved_objects.EntityDiscoveryApiKeyType.name]
    });
    return {
      clusterClient,
      soClient
    };
  }
});
exports.getApiKeyManager = getApiKeyManager;