"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.threatMappingEntriesAreValid = exports.createNamedAndClause = exports.createInnerAndClauses = exports.buildThreatMappingFilter = exports.buildEntriesMappingFilter = exports.MAX_CHUNK_SIZE = void 0;
var _fp = require("lodash/fp");
var _types = require("./types");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_CHUNK_SIZE = exports.MAX_CHUNK_SIZE = 1024;
const buildThreatMappingFilter = ({
  threatMappings,
  threatList,
  entryKey = 'value',
  allowedFieldsForTermsQuery
}) => {
  const query = buildEntriesMappingFilter({
    threatMappings,
    threatList,
    entryKey,
    allowedFieldsForTermsQuery
  });
  const filterChunk = {
    meta: {
      alias: null,
      negate: false,
      disabled: false
    },
    query
  };
  return filterChunk;
};

/**
 * Filters out any combined "AND" entries which do not include all the threat list items.
 */
exports.buildThreatMappingFilter = buildThreatMappingFilter;
const threatMappingEntriesAreValid = ({
  threatMappingEntries,
  threatListItem,
  entryKey
}) => threatMappingEntries.every(entry => {
  // DOES NOT MATCH clause allows undefined values
  if (entry.negate) {
    return true;
  }
  const itemValue = (0, _fp.get)(entry[entryKey], threatListItem.fields);
  return itemValue != null && itemValue.length === 1;
});
exports.threatMappingEntriesAreValid = threatMappingEntriesAreValid;
const createInnerAndClauses = ({
  threatMappingEntries,
  threatListItem,
  entryKey
}) => {
  return threatMappingEntries.reduce((accum, threatMappingEntry) => {
    const value = (0, _fp.get)(threatMappingEntry[entryKey], threatListItem.fields);
    const matchKey = threatMappingEntry[entryKey === 'field' ? 'value' : 'field'];
    if (threatMappingEntry.negate) {
      const negateClause = buildNegateClause(value, matchKey);
      if (negateClause) {
        accum.push(negateClause);
      }
    } else if (value != null && value.length === 1) {
      // These values could be potentially 10k+ large so mutating the array intentionally
      accum.push({
        match: {
          [matchKey]: {
            query: value[0]
          }
        }
      });
    }
    return accum;
  }, []);
};
exports.createInnerAndClauses = createInnerAndClauses;
const buildNegateClause = (value, matchKey) => {
  if (value == null) {
    return {
      exists: {
        field: matchKey
      }
    };
  }
  if (Array.isArray(value) && value.length === 1) {
    return {
      bool: {
        must_not: {
          match: {
            [matchKey]: {
              query: value[0]
            }
          }
        }
      }
    };
  }
  return undefined;
};
const createNamedAndClause = ({
  threatMappingEntries,
  threatListItem,
  entryKey,
  threatMappingIndex
}) => {
  const innerAndClauses = createInnerAndClauses({
    threatMappingEntries,
    threatListItem,
    entryKey
  });
  if (innerAndClauses.length !== 0) {
    // These values could be potentially 10k+ large so mutating the array intentionally
    return {
      bool: {
        _name: (0, _utils.encodeThreatMatchNamedQuery)({
          id: threatListItem._id,
          index: threatListItem._index,
          threatMappingIndex,
          queryType: _types.ThreatMatchQueryType.match
        }),
        filter: innerAndClauses
      }
    };
  }
};
exports.createNamedAndClause = createNamedAndClause;
const buildEntriesMappingFilter = ({
  threatMappings,
  threatList,
  entryKey,
  allowedFieldsForTermsQuery
}) => {
  const allFieldAllowedForTermQuery = entries => entries.every(entry => {
    var _allowedFieldsForTerm, _allowedFieldsForTerm2;
    return (allowedFieldsForTermsQuery === null || allowedFieldsForTermsQuery === void 0 ? void 0 : (_allowedFieldsForTerm = allowedFieldsForTermsQuery.source) === null || _allowedFieldsForTerm === void 0 ? void 0 : _allowedFieldsForTerm[entry.field]) && (allowedFieldsForTermsQuery === null || allowedFieldsForTermsQuery === void 0 ? void 0 : (_allowedFieldsForTerm2 = allowedFieldsForTermsQuery.threat) === null || _allowedFieldsForTerm2 === void 0 ? void 0 : _allowedFieldsForTerm2[entry.value]);
  });
  const combinedShould = threatMappings.reduce((acc, threatMapping, threatMappingIndex) => {
    if (threatMapping.entries.length > 1 || !allFieldAllowedForTermQuery(threatMapping.entries)) {
      threatList.forEach(threatListItem => {
        if (threatMappingEntriesAreValid({
          threatMappingEntries: threatMapping.entries,
          threatListItem,
          entryKey
        })) {
          const queryWithAndOrClause = createNamedAndClause({
            threatMappingEntries: threatMapping.entries,
            threatListItem,
            entryKey,
            threatMappingIndex
          });
          if (queryWithAndOrClause) {
            // These values can be 10k+ large, so using a push here for performance
            acc.push(queryWithAndOrClause);
          }
        }
      });
    } else {
      const threatMappingEntry = threatMapping.entries[0];
      const threats = threatList.map(threatListItem => (0, _fp.get)(threatMappingEntry[entryKey], threatListItem.fields)).filter(val => val).map(val => val[0]);
      if (threats.length > 0) {
        acc.push({
          terms: {
            _name: (0, _utils.encodeThreatMatchNamedQuery)({
              threatMappingIndex,
              queryType: _types.ThreatMatchQueryType.term
            }),
            [threatMappingEntry[entryKey === 'field' ? 'value' : 'field']]: threats
          }
        });
      }
    }
    return acc;
  }, []);
  return {
    bool: {
      should: combinedShould,
      minimum_should_match: 1
    }
  };
};
exports.buildEntriesMappingFilter = buildEntriesMappingFilter;