"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerUpdateInsightsRoute = void 0;
var _events = require("../../../lib/telemetry/event_based/events");
var _workflow_insights = require("../../../../common/api/endpoint/workflow_insights/workflow_insights");
var _services = require("../../services");
var _error_handler = require("../error_handler");
var _constants = require("../../../../common/endpoint/constants");
var _with_endpoint_authz = require("../with_endpoint_authz");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerUpdateInsightsRoute = (router, endpointContext) => {
  router.versioned.put({
    access: 'internal',
    path: _constants.WORKFLOW_INSIGHTS_UPDATE_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '1',
    validate: {
      request: _workflow_insights.UpdateWorkflowInsightRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canReadWorkflowInsights']
  }, endpointContext.logFactory.get('workflowInsights'), updateInsightsRouteHandler(endpointContext)));
};
exports.registerUpdateInsightsRoute = registerUpdateInsightsRoute;
const updateInsightsRouteHandler = endpointContext => {
  const logger = endpointContext.logFactory.get('workflowInsights');
  const isOnlyActionTypeUpdate = body => {
    var _body$action;
    // Type guard is done by schema validation
    if (!(body !== null && body !== void 0 && (_body$action = body.action) !== null && _body$action !== void 0 && _body$action.type)) return false;
    // Make sure the body only contains the action.type field
    return Object.keys(body).length === 1 && Object.keys(body.action).length === 1;
  };
  return async (context, request, response) => {
    const {
      insightId
    } = request.params;
    const {
      canWriteWorkflowInsights
    } = await endpointContext.service.getEndpointAuthz(request);
    const {
      endpointManagementSpaceAwarenessEnabled,
      defendInsightsPolicyResponseFailure
    } = endpointContext.experimentalFeatures;
    if (request.body.type === 'policy_response_failure' && !defendInsightsPolicyResponseFailure) {
      return response.badRequest({
        body: 'policy_response_failure insight type requires defendInsightsPolicyResponseFailure feature flag'
      });
    }
    const onlyActionTypeUpdate = isOnlyActionTypeUpdate(request.body);
    if (!canWriteWorkflowInsights && !onlyActionTypeUpdate) {
      return response.forbidden({
        body: 'Unauthorized to update workflow insights'
      });
    }
    if (onlyActionTypeUpdate) {
      var _request$body$action;
      if (((_request$body$action = request.body.action) === null || _request$body$action === void 0 ? void 0 : _request$body$action.type) === 'remediated') {
        const telemetry = endpointContext.service.getTelemetryService();
        telemetry.reportEvent(_events.ENDPOINT_WORKFLOW_INSIGHTS_REMEDIATED_EVENT.eventType, {
          insightId
        });
      }
    }
    logger.debug(`Updating insight ${insightId}`);
    try {
      const retrievedInsight = (await _services.securityWorkflowInsightsService.fetch({
        ids: [insightId]
      }))[0];
      if (!retrievedInsight) {
        throw new Error('Failed to retrieve insight');
      }
      const backingIndex = retrievedInsight._index;

      // If the endpoint management space awareness feature is enabled, we need to ensure that the agent IDs are in the current space
      if (endpointManagementSpaceAwarenessEnabled) {
        var _retrievedInsight$_so, _retrievedInsight$_so2, _request$body$target;
        const spaceId = (await context.securitySolution).getSpaceId();
        const fleetServices = endpointContext.service.getInternalFleetServices(spaceId);

        // We need to make sure the agent IDs, both existing and injected through the request body, are in the current space
        const existingAgentIds = retrievedInsight === null || retrievedInsight === void 0 ? void 0 : (_retrievedInsight$_so = retrievedInsight._source) === null || _retrievedInsight$_so === void 0 ? void 0 : (_retrievedInsight$_so2 = _retrievedInsight$_so.target) === null || _retrievedInsight$_so2 === void 0 ? void 0 : _retrievedInsight$_so2.ids;
        const newAgentIds = (_request$body$target = request.body.target) === null || _request$body$target === void 0 ? void 0 : _request$body$target.ids;
        const combinedAgentIds = Array.from(new Set([...(existingAgentIds !== null && existingAgentIds !== void 0 ? existingAgentIds : []), ...(newAgentIds !== null && newAgentIds !== void 0 ? newAgentIds : [])]));
        if (combinedAgentIds.length > 0) {
          await fleetServices.ensureInCurrentSpace({
            agentIds: combinedAgentIds
          });
        }
      }
      const body = await _services.securityWorkflowInsightsService.update(insightId, request.body, backingIndex);
      return response.ok({
        body
      });
    } catch (e) {
      return (0, _error_handler.errorHandler)(logger, response, e);
    }
  };
};