"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createConfig = exports.configSchema = void 0;
var _configSchema = require("@kbn/config-schema");
var _constants = require("../common/constants");
var _experimental_features = require("../common/experimental_features");
var _config_settings = require("../common/config_settings");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Validates if the value provided is a valid duration for use with Task Manager (ex. 5m, 4s)
 */
const isValidTaskManagerDuration = value => {
  if (!/^\d+[s,m]{1}$/.test(value)) {
    return `Invalid duration [${value}]. Value must be a number followed by either 's' for seconds or 'm' for minutes `;
  }
};
const configSchema = exports.configSchema = _configSchema.schema.object({
  maxRuleImportExportSize: _configSchema.schema.number({
    defaultValue: 10000
  }),
  maxRuleImportPayloadBytes: _configSchema.schema.number({
    defaultValue: 10485760
  }),
  maxTimelineImportExportSize: _configSchema.schema.number({
    defaultValue: 10000
  }),
  maxTimelineImportPayloadBytes: _configSchema.schema.number({
    defaultValue: 10485760
  }),
  /**
   * This is used within the merge strategies:
   * server/lib/detection_engine/rule_types/utils/source_fields_merging
   *
   * For determining which strategy for merging "fields" and "_source" together to get
   * runtime fields, constant keywords, etc...
   *
   * "missingFields" (default) This will only merge fields that are missing from the _source and exist in the fields.
   * "noFields" This will turn off all merging of runtime fields, constant keywords from fields.
   * "allFields" This will merge and overwrite anything found within "fields" into "_source" before indexing the data.
   */
  alertMergeStrategy: _configSchema.schema.oneOf([_configSchema.schema.literal('allFields'), _configSchema.schema.literal('missingFields'), _configSchema.schema.literal('noFields')], {
    defaultValue: 'missingFields'
  }),
  /**
   * This is used within the merge strategies:
   * server/lib/detection_engine/rule_types/utils/source_fields_merging
   *
   * For determining if we need to ignore particular "fields" and not merge them with "_source" such as
   * runtime fields, constant keywords, etc...
   *
   * This feature and functionality is mostly as "safety feature" meaning that we have had bugs in the past
   * where something down the stack unexpectedly ends up in the fields API which causes documents to not
   * be indexable. Rather than changing alertMergeStrategy to be "noFields", you can use this array to add
   * any problematic values.
   *
   * You can use plain dotted notation strings such as "host.name" or a regular expression such as "/host\..+/"
   */
  alertIgnoreFields: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
    defaultValue: [],
    validate(ignoreFields) {
      const errors = ignoreFields.flatMap((ignoreField, index) => {
        if (ignoreField.startsWith('/') && ignoreField.endsWith('/')) {
          try {
            new RegExp(ignoreField.slice(1, -1));
            return [];
          } catch (error) {
            return [`"${error.message}" at array position ${index}`];
          }
        } else {
          return [];
        }
      });
      if (errors.length !== 0) {
        return errors.join('. ');
      } else {
        return undefined;
      }
    }
  }),
  [_constants.SIGNALS_INDEX_KEY]: _configSchema.schema.string({
    defaultValue: _constants.DEFAULT_SIGNALS_INDEX
  }),
  /**
   * For internal use. A list of string values (comma delimited) that will enable experimental
   * type of functionality that is not yet released. Valid values for this settings need to
   * be defined in:
   * `x-pack/solutions/security/plugins/security_solution/common/experimental_features.ts`
   * under the `allowedExperimentalValues` object
   *
   * @example
   * xpack.securitySolution.enableExperimental:
   *   - someCrazyFeature
   *   - someEvenCrazierFeature
   */
  enableExperimental: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
    defaultValue: () => []
  }),
  /**
   * Endpoint Artifacts Configuration: the interval between runs of the task that builds the
   * artifacts and associated manifest.
   */
  packagerTaskInterval: _configSchema.schema.string({
    defaultValue: '60s'
  }),
  /**
   * Endpoint Artifacts Configuration: timeout value for how long the task should run.
   */
  packagerTaskTimeout: _configSchema.schema.string({
    defaultValue: '20m'
  }),
  /**
   * Artifacts Configuration for package policy update concurrency
   */
  packagerTaskPackagePolicyUpdateBatchSize: _configSchema.schema.number({
    defaultValue: 25,
    max: 50,
    min: 1
  }),
  /**
   * Complete External Response Actions task: interval duration
   */
  completeExternalResponseActionsTaskInterval: _configSchema.schema.string({
    defaultValue: '60s',
    validate: isValidTaskManagerDuration
  }),
  /**
   * Complete External Response Actions task: Timeout value for how long the task should run
   */
  completeExternalResponseActionsTaskTimeout: _configSchema.schema.string({
    defaultValue: '5m',
    validate: isValidTaskManagerDuration
  }),
  /**
   * For internal use. Specify which version of the Detection Rules fleet package to install
   * when upgrading rules. If not provided, the latest compatible package will be installed,
   * or if running from a dev environment or -SNAPSHOT build, the latest pre-release package
   * will be used (if fleet is available or not within an airgapped environment).
   *
   * Note: This is for `upgrade only`, which occurs by means of the `useUpgradeSecurityPackages`
   * hook when navigating to a Security Solution page. The package version specified in
   * `fleet_packages.json` in project root will always be installed first on Kibana start if
   * the package is not already installed.
   */
  prebuiltRulesPackageVersion: _configSchema.schema.maybe(_configSchema.schema.string()),
  enabled: _configSchema.schema.boolean({
    defaultValue: true
  }),
  enableUiSettingsValidations: _configSchema.schema.boolean({
    defaultValue: false
  }),
  /**
   * The Max number of Bytes allowed for the `upload` endpoint response action
   */
  maxUploadResponseActionFileBytes: _configSchema.schema.number({
    defaultValue: 26214400,
    // 25MB,
    max: 104857600 // 100MB,
  }),
  /**
   * Disables the auto-install/enable of the Elastic Defend SIEM rule.
   * Whenever a Policy is created via Fleet's API, we check if the corresponding Elastic Defend SIEM
   * rule is installed/enabled in the active space, and if not, we auto-install it. Set this configuration
   * setting to `false` to disable that behavior
   */
  disableEndpointRuleAutoInstall: _configSchema.schema.boolean({
    defaultValue: false
  }),
  /**
   * Defines the settings for a specific offering of the Security Solution app.
   * They override the default values.
   * @example
   * xpack.securitySolution.offeringSettings: {
   *  "ILMEnabled": false,
   * }
   */
  offeringSettings: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.boolean(), {
    defaultValue: {}
  }),
  entityAnalytics: _configSchema.schema.object({
    riskEngine: _configSchema.schema.object({
      alertSampleSizePerShard: _configSchema.schema.number({
        defaultValue: 10_000
      })
    }),
    assetCriticality: _configSchema.schema.object({
      csvUpload: _configSchema.schema.object({
        errorRetries: _configSchema.schema.number({
          defaultValue: 1
        }),
        maxBulkRequestBodySizeBytes: _configSchema.schema.number({
          defaultValue: 100_000
        }) // 100KB
      })
    }),
    entityStore: _configSchema.schema.object({
      syncDelay: _configSchema.schema.duration({
        defaultValue: '60s'
      }),
      frequency: _configSchema.schema.duration({
        defaultValue: '60s'
      }),
      developer: _configSchema.schema.object({
        pipelineDebugMode: _configSchema.schema.boolean({
          defaultValue: false
        })
      })
    }),
    monitoring: _configSchema.schema.object({
      privileges: _configSchema.schema.object({
        users: _configSchema.schema.object({
          maxPrivilegedUsersAllowed: _configSchema.schema.number({
            defaultValue: 10000
          }),
          csvUpload: _configSchema.schema.object({
            errorRetries: _configSchema.schema.number({
              defaultValue: 1
            }),
            maxBulkRequestBodySizeBytes: _configSchema.schema.number({
              defaultValue: 100_000
            }) // 100KB
          })
        })
      })
    })
  }),
  siemRuleMigrations: _configSchema.schema.maybe(_configSchema.schema.object({
    elserInferenceId: _configSchema.schema.maybe(_configSchema.schema.string())
  })),
  cdn: _configSchema.schema.maybe(_configSchema.schema.object({
    url: _configSchema.schema.maybe(_configSchema.schema.string()),
    // PEM-encoded public key used to verify the global artifact manifest signature.
    publicKey: _configSchema.schema.maybe(_configSchema.schema.string())
  })),
  telemetry: _configSchema.schema.maybe(_configSchema.schema.object({
    queryConfig: _configSchema.schema.object({
      pageSize: _configSchema.schema.maybe(_configSchema.schema.number()),
      maxResponseSize: _configSchema.schema.maybe(_configSchema.schema.number()),
      maxCompressedResponseSize: _configSchema.schema.maybe(_configSchema.schema.number())
    })
  }))
});
const createConfig = context => {
  const pluginConfig = context.config.get();
  const logger = context.logger.get('config');
  const {
    invalid,
    features: experimentalFeatures
  } = (0, _experimental_features.parseExperimentalConfigValue)(pluginConfig.enableExperimental);
  if (invalid.length) {
    logger.warn(`Unsupported "xpack.securitySolution.enableExperimental" values detected.
The following configuration values are no longer supported and should be removed from the kibana configuration file:

    xpack.securitySolution.enableExperimental:
${invalid.map(key => `      - ${key}`).join('\n')}
`);
  }
  const {
    invalid: invalidConfigSettings,
    settings
  } = (0, _config_settings.parseConfigSettings)(pluginConfig.offeringSettings);
  if (invalidConfigSettings.length) {
    logger.warn(`Unsupported "xpack.securitySolution.offeringSettings" values detected.
The following configuration values are no longer supported and should be removed from the kibana configuration file:
${invalidConfigSettings.map(key => `      - ${key}`).join('\n')}
`);
  }
  return {
    ...pluginConfig,
    experimentalFeatures,
    settings
  };
};
exports.createConfig = createConfig;