"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ensureTimelineIsSaved = ensureTimelineIsSaved;
exports.extractTimelineIdsAndVersions = extractTimelineIdsAndVersions;
exports.isHttpFetchError = isHttpFetchError;
exports.refreshTimelines = refreshTimelines;
var _uuid = require("uuid");
var _all = require("../../containers/all");
var _inputs = require("../../../common/store/inputs");
var _actions = require("../actions");
var _timeline = require("../../../../common/api/timeline");
var _selectors = require("../selectors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Refreshes all timelines, so changes are propagated to everywhere on the page
 */
function refreshTimelines(state) {
  const allTimelineQuery = _inputs.inputsSelectors.globalQueryByIdSelector()(state, _all.ALL_TIMELINE_QUERY_ID);
  if (allTimelineQuery.refetch != null) {
    allTimelineQuery.refetch();
  }
}

/**
 * Given a timeline model, it will return that model when the timeline has been saved before,
 * or save a draft version of that timeline.
 * This is a usefull check for when you're working with timeline-associated saved objects
 * which require the exitence of a timeline's `savedObjectId`.
 */
async function ensureTimelineIsSaved({
  localTimelineId,
  timeline,
  store
}) {
  // In case `savedObjectId` exists, the timeline has been saved before.
  if (timeline.savedObjectId) {
    return timeline;
  }

  // The timeline hasn't been saved, so let's create make it a draft.
  await store.dispatch((0, _actions.updateTimeline)({
    id: localTimelineId,
    timeline: {
      ...timeline,
      status: _timeline.TimelineStatusEnum.draft
    }
  }));

  // The draft needs to be persisted
  await store.dispatch((0, _actions.saveTimeline)({
    id: localTimelineId,
    saveAsNew: false
  }));

  // Make sure we're returning the most updated version of the timeline
  return (0, _selectors.selectTimelineById)(store.getState(), localTimelineId);
}
function isHttpFetchError(error) {
  return error !== null && typeof error === 'object' && 'body' in error && error.body !== null && typeof error.body === 'object' && `status_code` in error.body && typeof error.body.status_code === 'number';
}
function extractTimelineIdsAndVersions(timeline) {
  var _timeline$savedObject;
  // When a timeline hasn't been saved yet, its `savedObectId` is not defined.
  // In that case, we want to overwrite all locally created properties for the
  // timeline id, the timeline template id and the timeline template version.
  //

  let templateTimelineId = timeline.savedObjectId ? timeline.templateTimelineId : null;

  // if not templateTimelineId is given and we know that the timelineType is `template`,
  // we should generate one to handle template duplicate case
  if (!templateTimelineId && timeline.timelineType === 'template') {
    templateTimelineId = (0, _uuid.v4)();
  }
  return {
    timelineId: (_timeline$savedObject = timeline.savedObjectId) !== null && _timeline$savedObject !== void 0 ? _timeline$savedObject : null,
    timelineVersion: timeline.version,
    templateTimelineId,
    templateTimelineVersion: timeline.savedObjectId ? timeline.templateTimelineVersion : null
  };
}