"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.sourceDestinationFieldMappings = exports.lmc = exports.getSourceLayer = exports.getRequiredMapsFields = exports.getLineLayer = exports.getLayerList = exports.getDestinationLayer = exports.SUM_OF_SOURCE_BYTES = exports.SUM_OF_SERVER_BYTES = exports.SUM_OF_DESTINATION_BYTES = exports.SUM_OF_CLIENT_BYTES = void 0;
var _uuid = require("uuid");
var _eui = require("@elastic/eui");
var _common = require("@kbn/maps-plugin/common");
var i18n = _interopRequireWildcard(require("./translations"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Update field mappings to modify what fields will be returned to map tooltip
const sourceFieldMappings = {
  'host.name': i18n.HOST,
  'source.ip': i18n.SOURCE_IP,
  'source.domain': i18n.SOURCE_DOMAIN,
  'source.geo.country_iso_code': i18n.LOCATION,
  'source.as.organization.name': i18n.ASN
};
const destinationFieldMappings = {
  'host.name': i18n.HOST,
  'destination.ip': i18n.DESTINATION_IP,
  'destination.domain': i18n.DESTINATION_DOMAIN,
  'destination.geo.country_iso_code': i18n.LOCATION,
  'destination.as.organization.name': i18n.ASN
};
const clientFieldMappings = {
  'host.name': i18n.HOST,
  'client.ip': i18n.CLIENT_IP,
  'client.domain': i18n.CLIENT_DOMAIN,
  'client.geo.country_iso_code': i18n.LOCATION,
  'client.as.organization.name': i18n.ASN
};
const serverFieldMappings = {
  'host.name': i18n.HOST,
  'server.ip': i18n.SERVER_IP,
  'server.domain': i18n.SERVER_DOMAIN,
  'server.geo.country_iso_code': i18n.LOCATION,
  'server.as.organization.name': i18n.ASN
};

// Mapping of field -> display name for use within map tooltip
const sourceDestinationFieldMappings = exports.sourceDestinationFieldMappings = {
  ...sourceFieldMappings,
  ...destinationFieldMappings,
  ...clientFieldMappings,
  ...serverFieldMappings
};

// Field names of LineLayer props returned from Maps API
const SUM_OF_SOURCE_BYTES = exports.SUM_OF_SOURCE_BYTES = 'sum_of_source.bytes';
const SUM_OF_DESTINATION_BYTES = exports.SUM_OF_DESTINATION_BYTES = 'sum_of_destination.bytes';
const SUM_OF_CLIENT_BYTES = exports.SUM_OF_CLIENT_BYTES = 'sum_of_client.bytes';
const SUM_OF_SERVER_BYTES = exports.SUM_OF_SERVER_BYTES = 'sum_of_server.bytes';
const APM_LAYER_FIELD_MAPPING = {
  source: {
    metricField: 'client.bytes',
    geoField: 'client.geo.location',
    tooltipProperties: Object.keys(clientFieldMappings),
    label: i18n.CLIENT_LAYER
  },
  destination: {
    metricField: 'server.bytes',
    geoField: 'server.geo.location',
    tooltipProperties: Object.keys(serverFieldMappings),
    label: i18n.SERVER_LAYER
  }
};

// Mapping to fields for creating specific layers for a given index pattern
// e.g. The apm-* index pattern needs layers for client/server instead of source/destination
const lmc = exports.lmc = {
  default: {
    source: {
      metricField: 'source.bytes',
      geoField: 'source.geo.location',
      tooltipProperties: Object.keys(sourceFieldMappings),
      label: i18n.SOURCE_LAYER
    },
    destination: {
      metricField: 'destination.bytes',
      geoField: 'destination.geo.location',
      tooltipProperties: Object.keys(destinationFieldMappings),
      label: i18n.DESTINATION_LAYER
    }
  },
  'apm-*': APM_LAYER_FIELD_MAPPING,
  'traces-apm*,logs-apm*,metrics-apm*,apm-*': APM_LAYER_FIELD_MAPPING
};
const getRequiredMapsFields = title => {
  var _lmc$title;
  const fieldMappings = (_lmc$title = lmc[title]) !== null && _lmc$title !== void 0 ? _lmc$title : lmc.default;
  return [fieldMappings.source.metricField, fieldMappings.source.geoField, fieldMappings.destination.metricField, fieldMappings.destination.geoField];
};
exports.getRequiredMapsFields = getRequiredMapsFields;
/**
 * Returns `Source/Destination Point-to-point` Map LayerList configuration, with a source,
 * destination, and line layer for each of the provided indexPatterns
 *
 * @param dependencies dependencies, such as theme, necessary to configure layers
 * @param indexPatternIds array of indexPatterns' title and id
 */
const getLayerList = (dependencies, indexPatternIds) => {
  return [...indexPatternIds.reduce((acc, {
    title,
    id
  }) => {
    var _lmc$title2, _lmc$title$destinatio, _lmc$title3, _lmc$title$source, _lmc$title4;
    const layerGroupDescriptor = {
      id: (0, _uuid.v4)(),
      label: title,
      sourceDescriptor: null,
      type: _common.LAYER_TYPE.LAYER_GROUP,
      visible: true
    };
    return [...acc, getLineLayer(title, id, layerGroupDescriptor.id, (_lmc$title2 = lmc[title]) !== null && _lmc$title2 !== void 0 ? _lmc$title2 : lmc.default), getDestinationLayer(dependencies, title, id, layerGroupDescriptor.id, (_lmc$title$destinatio = (_lmc$title3 = lmc[title]) === null || _lmc$title3 === void 0 ? void 0 : _lmc$title3.destination) !== null && _lmc$title$destinatio !== void 0 ? _lmc$title$destinatio : lmc.default.destination), getSourceLayer(dependencies, title, id, layerGroupDescriptor.id, (_lmc$title$source = (_lmc$title4 = lmc[title]) === null || _lmc$title4 === void 0 ? void 0 : _lmc$title4.source) !== null && _lmc$title$source !== void 0 ? _lmc$title$source : lmc.default.source), layerGroupDescriptor];
  }, [])];
};

/**
 * Returns Document Data Source layer configuration ('source.geo.location') for the given
 * indexPattern title/id
 *
 * @param indexPatternTitle used as layer name in LayerToC UI: "${indexPatternTitle} | Source point"
 * @param indexPatternId used as layer's indexPattern to query for data
 * @param parentId
 * @param layerDetails layer-specific field details
 */
exports.getLayerList = getLayerList;
const getSourceLayer = (dependencies, indexPatternTitle, indexPatternId, parentId, layerDetails) => {
  return {
    sourceDescriptor: {
      id: (0, _uuid.v4)(),
      type: 'ES_SEARCH',
      applyGlobalQuery: true,
      geoField: layerDetails.geoField,
      filterByMapBounds: false,
      tooltipProperties: layerDetails.tooltipProperties,
      useTopHits: false,
      topHitsTimeField: '@timestamp',
      topHitsSize: 1,
      indexPatternId
    },
    style: {
      type: 'VECTOR',
      properties: {
        fillColor: {
          type: 'STATIC',
          options: {
            color: dependencies.euiTheme.colors.vis.euiColorVis4
          }
        },
        lineColor: {
          type: 'STATIC',
          options: {
            color: dependencies.euiTheme.colors.vis.euiColorVisBase0
          }
        },
        lineWidth: {
          type: 'STATIC',
          options: {
            size: 2
          }
        },
        iconSize: {
          type: 'STATIC',
          options: {
            size: 8
          }
        },
        iconOrientation: {
          type: 'STATIC',
          options: {
            orientation: 0
          }
        },
        symbolizeAs: {
          options: {
            value: 'icon'
          }
        },
        icon: {
          type: 'STATIC',
          options: {
            value: 'home'
          }
        }
      }
    },
    id: (0, _uuid.v4)(),
    parent: parentId,
    label: `${indexPatternTitle} | ${layerDetails.label}`,
    minZoom: 0,
    maxZoom: 24,
    alpha: 1,
    visible: true,
    type: _common.LAYER_TYPE.GEOJSON_VECTOR,
    query: {
      query: '',
      language: 'kuery'
    },
    joins: []
  };
};

/**
 * Returns Document Data Source layer configuration ('destination.geo.location') for the given
 * indexPattern title/id
 *
 * @param indexPatternTitle used as layer name in LayerToC UI: "${indexPatternTitle} | Destination point"
 * @param indexPatternId used as layer's indexPattern to query for data
 * @param parentId used as layer's indexPattern to query for data
 * @param layerDetails layer-specific field details
 *
 */
exports.getSourceLayer = getSourceLayer;
const getDestinationLayer = (dependencies, indexPatternTitle, indexPatternId, parentId, layerDetails) => {
  return {
    sourceDescriptor: {
      id: (0, _uuid.v4)(),
      type: 'ES_SEARCH',
      scalingType: _common.SCALING_TYPES.LIMIT,
      applyGlobalQuery: true,
      geoField: layerDetails.geoField,
      filterByMapBounds: true,
      tooltipProperties: layerDetails.tooltipProperties,
      useTopHits: false,
      topHitsTimeField: '@timestamp',
      topHitsSize: 1,
      indexPatternId
    },
    style: {
      type: 'VECTOR',
      properties: {
        fillColor: {
          type: 'STATIC',
          options: {
            color: dependencies.euiTheme.colors.vis.euiColorVis2
          }
        },
        lineColor: {
          type: 'STATIC',
          options: {
            color: dependencies.euiTheme.colors.vis.euiColorVisBase0
          }
        },
        lineWidth: {
          type: 'STATIC',
          options: {
            size: 2
          }
        },
        iconSize: {
          type: 'STATIC',
          options: {
            size: 8
          }
        },
        iconOrientation: {
          type: 'STATIC',
          options: {
            orientation: 0
          }
        },
        symbolizeAs: {
          options: {
            value: 'icon'
          }
        },
        icon: {
          type: 'STATIC',
          options: {
            value: 'marker'
          }
        }
      }
    },
    id: (0, _uuid.v4)(),
    parent: parentId,
    label: `${indexPatternTitle} | ${layerDetails.label}`,
    minZoom: 0,
    maxZoom: 24,
    alpha: 1,
    visible: true,
    type: _common.LAYER_TYPE.GEOJSON_VECTOR,
    query: {
      query: '',
      language: 'kuery'
    }
  };
};

/**
 * Returns Point-to-point Data Source layer configuration ('source.geo.location' &
 * 'source.geo.location') for the given indexPattern title/id
 *
 * @param indexPatternTitle used as layer name in LayerToC UI: "${indexPatternTitle} | Line"
 * @param indexPatternId used as layer's indexPattern to query for data
 * @param parentId
 * @param layerDetails layer-specific field details
 */
exports.getDestinationLayer = getDestinationLayer;
const getLineLayer = (indexPatternTitle, indexPatternId, parentId, layerDetails) => {
  const euiVisColorPalette = (0, _eui.euiPaletteColorBlind)();
  return {
    sourceDescriptor: {
      type: _common.SOURCE_TYPES.ES_PEW_PEW,
      applyGlobalQuery: true,
      id: (0, _uuid.v4)(),
      indexPatternId,
      sourceGeoField: layerDetails.source.geoField,
      destGeoField: layerDetails.destination.geoField,
      metrics: [{
        type: 'sum',
        field: layerDetails.source.metricField,
        label: layerDetails.source.metricField
      }, {
        type: 'sum',
        field: layerDetails.destination.metricField,
        label: layerDetails.destination.metricField
      }]
    },
    style: {
      type: 'VECTOR',
      properties: {
        fillColor: {
          type: 'STATIC',
          options: {
            color: euiVisColorPalette[1]
          }
        },
        lineColor: {
          type: 'STATIC',
          options: {
            color: euiVisColorPalette[1]
          }
        },
        lineWidth: {
          type: 'DYNAMIC',
          options: {
            field: {
              label: 'count',
              name: 'doc_count',
              origin: 'source'
            },
            minSize: 1,
            maxSize: 8,
            fieldMetaOptions: {
              isEnabled: true,
              sigma: 3
            }
          }
        },
        iconSize: {
          type: 'STATIC',
          options: {
            size: 10
          }
        },
        iconOrientation: {
          type: 'STATIC',
          options: {
            orientation: 0
          }
        },
        symbolizeAs: {
          options: {
            value: 'icon'
          }
        },
        icon: {
          type: 'STATIC',
          options: {
            value: 'airfield'
          }
        }
      }
    },
    id: (0, _uuid.v4)(),
    parent: parentId,
    label: `${indexPatternTitle} | ${i18n.LINE_LAYER}`,
    minZoom: 0,
    maxZoom: 24,
    alpha: 0.5,
    visible: true,
    type: _common.LAYER_TYPE.GEOJSON_VECTOR,
    query: {
      query: '',
      language: 'kuery'
    }
  };
};
exports.getLineLayer = getLineLayer;