"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.hasCapabilities = exports.existCapabilities = void 0;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * The format of defining features supports OR and AND mechanism. To specify features in an OR fashion
 * they can be defined in a single level array like: [requiredFeature1, requiredFeature2]. If either of these features
 * is satisfied the link would be included. To require that the features be AND'd together a second level array
 * can be specified: [feature1, [feature2, feature3]] this would result in feature1 || (feature2 && feature3). To specify
 * features that all must be and'd together an example would be: [[feature1, feature2]], this would result in the boolean
 * operation feature1 && feature2.
 *
 * The final format is to specify a single feature, this would be like: features: feature1, which is the same as
 * features: [feature1]
 */

/**
 * Checks if the capabilities exist and are enabled in the capabilities object.
 * It checks if the capabilities are enabled or not.
 * It does not distinguish between missing and disabled capabilities.
 */
const hasCapabilities = (capabilities, requiredCapabilities) => {
  if (!requiredCapabilities || requiredCapabilities.length === 0) {
    return true;
  }
  if (!(0, _lodash.isArray)(requiredCapabilities)) {
    return !!(0, _lodash.get)(capabilities, requiredCapabilities, false);
  } else {
    return requiredCapabilities.some(requiredCapsOr => {
      if ((0, _lodash.isArray)(requiredCapsOr)) {
        return requiredCapsOr.every(requiredCapsAnd => (0, _lodash.get)(capabilities, requiredCapsAnd, false));
      }
      return (0, _lodash.get)(capabilities, requiredCapsOr, false);
    });
  }
};

/**
 * Checks if the capabilities exist in the capabilities object.
 * It does not check if the capabilities are enabled or not.
 * This is used to check if the capabilities have been registered in the Kibana feature privileges configuration.
 */
exports.hasCapabilities = hasCapabilities;
const existCapabilities = (capabilities, requiredCapabilities) => {
  if (!requiredCapabilities || requiredCapabilities.length === 0) {
    return true;
  }
  if (!(0, _lodash.isArray)(requiredCapabilities)) {
    return (0, _lodash.get)(capabilities, requiredCapabilities) != null;
  } else {
    return requiredCapabilities.some(requiredCapsOr => {
      if ((0, _lodash.isArray)(requiredCapsOr)) {
        return requiredCapsOr.every(requiredCapsAnd => (0, _lodash.get)(capabilities, requiredCapsAnd) != null);
      }
      return (0, _lodash.get)(capabilities, requiredCapsOr) != null;
    });
  }
};
exports.existCapabilities = existCapabilities;