"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.wrapErrorIfNeeded = exports.wrapErrorAndRejectPromise = exports.retryOnError = exports.mergeAndAppendArrays = exports.getElapsedTime = exports.fetchActiveSpaceId = exports.createToolingLogger = exports.RETRYABLE_TRANSIENT_ERRORS = exports.EndpointDataLoadingError = void 0;
var _lodash = require("lodash");
var _toolingLog = require("@kbn/tooling-log");
var _moment = _interopRequireDefault(require("moment/moment"));
var _format_axios_error = require("../format_axios_error");
var _errors = require("../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const RETRYABLE_TRANSIENT_ERRORS = exports.RETRYABLE_TRANSIENT_ERRORS = ['no_shard_available_action_exception', 'illegal_index_shard_state_exception'];
class EndpointDataLoadingError extends _errors.EndpointError {
  constructor(message, meta) {
    super(message);
    this.meta = meta;
  }
}
exports.EndpointDataLoadingError = EndpointDataLoadingError;
const wrapErrorIfNeeded = error => error instanceof EndpointDataLoadingError ? error : new EndpointDataLoadingError(error.message, error);

// Use it in Promise's `.catch()` as `.catch(wrapErrorAndRejectPromise)`
exports.wrapErrorIfNeeded = wrapErrorIfNeeded;
const wrapErrorAndRejectPromise = error => Promise.reject(wrapErrorIfNeeded(error));
exports.wrapErrorAndRejectPromise = wrapErrorAndRejectPromise;
const mergeAndAppendArrays = (destinationObj, srcObj) => {
  const customizer = (objValue, srcValue) => {
    if (Array.isArray(objValue)) {
      return objValue.concat(srcValue);
    }
  };
  return (0, _lodash.mergeWith)(destinationObj, srcObj, customizer);
};

/**
 * Call the provided `callback` and retry that call if it fails and the error in the failure
 * contains one of the `errors` provided.
 * @param callback
 * @param errors
 * @param tryCount
 * @param interval
 * @param logger
 */
exports.mergeAndAppendArrays = mergeAndAppendArrays;
const retryOnError = async (callback, errors, logger, tryCount = 5, interval = 10000) => {
  const log = logger !== null && logger !== void 0 ? logger : createToolingLogger('silent');
  const msg = message => `retryOnError(): ${message}`;
  const isRetryableError = err => {
    return errors.some(retryMessage => {
      if (typeof retryMessage === 'string') {
        return err.message.includes(retryMessage);
      } else {
        return retryMessage.test(err.message);
      }
    });
  };
  log.indent(4);
  let attempt = 1;
  let responsePromise;
  while (attempt <= tryCount) {
    const thisAttempt = attempt;
    attempt++;
    log.debug(msg(`attempt ${thisAttempt} started at: ${new Date().toISOString()}`));
    try {
      responsePromise = callback(); // store promise so that if it fails and no more attempts, we return the last failure
      const result = await responsePromise;
      log.debug(msg(`attempt ${thisAttempt} was successful. Exiting retry`));
      log.indent(-4);
      return result;
    } catch (err) {
      log.warning(msg(`attempt ${thisAttempt} failed with: ${err.message.split('\n').at(0)}`));
      log.verbose(err);

      // If not an error that is retryable, then end loop here and return that error;
      if (!isRetryableError(err)) {
        log.error(err);
        log.error(msg('non-retryable error encountered'));
        log.indent(-4);
        return Promise.reject(err);
      }
    }
    await new Promise(resolve => setTimeout(resolve, interval));
  }
  log.error(msg(`max retry attempts reached. returning last failure`));
  log.indent(-4);

  // Last resort: return the last rejected Promise.
  // @ts-expect-error TS2454: Variable 'responsePromise' is used before being assigned.
  return responsePromise;
};
exports.retryOnError = retryOnError;
/**
 * Creates an instance of `ToolingLog` that outputs to `stdout`.
 * The default log `level` for all instances can be set by setting the function's `defaultLogLevel`
 * property. Default logging level can also be set from CLI scripts that use the `@kbn/dev-cli-runner`
 * by calling the `setDefaultLogLevelFromCliFlags(flags)` and passing in the `flags` property.
 *
 * @param level
 *
 * @example
 * // Set default log level - example: from cypress for CI jobs
 * createLogger.defaultLogLevel = 'verbose'
 */
const createToolingLogger = level => {
  return new _toolingLog.ToolingLog({
    level: level || createToolingLogger.defaultLogLevel,
    writeTo: process.stdout
  });
};
exports.createToolingLogger = createToolingLogger;
createToolingLogger.defaultLogLevel = 'info';
createToolingLogger.setDefaultLogLevelFromCliFlags = flags => {
  createToolingLogger.defaultLogLevel = flags.verbose ? 'verbose' : flags.debug ? 'debug' : flags.silent ? 'silent' : flags.quiet ? 'error' : 'info';
};

/**
 * Get human readable string of time elapsed between to dates. Return value will be in the format
 * of `hh:mm:ss.ms`
 * @param startDate
 * @param endTime
 */
const getElapsedTime = (startDate, endTime = new Date()) => {
  const durationObj = _moment.default.duration((0, _moment.default)(endTime).diff(startDate));
  const pad = (num, max = 2) => {
    return String(num).padStart(max, '0');
  };
  const hours = pad(durationObj.hours());
  const minutes = pad(durationObj.minutes());
  const seconds = pad(durationObj.seconds());
  const milliseconds = pad(durationObj.milliseconds(), 3);
  return `${hours}:${minutes}:${seconds}.${milliseconds}`;
};
exports.getElapsedTime = getElapsedTime;
const fetchActiveSpaceId = exports.fetchActiveSpaceId = (0, _lodash.memoize)(async kbnClient => {
  return kbnClient.request({
    method: 'GET',
    path: `/internal/spaces/_active_space`
  }).catch(_format_axios_error.catchAxiosErrorFormatAndThrow).then(response => response.data.id);
});