"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KQL_FILTER_NOT_CUSTOMIZED_RULES = exports.KQL_FILTER_MUTABLE_RULES = exports.KQL_FILTER_IMMUTABLE_RULES = exports.KQL_FILTER_ENABLED_RULES = exports.KQL_FILTER_DISABLED_RULES = exports.KQL_FILTER_CUSTOMIZED_RULES = void 0;
exports.convertRuleSearchTermToKQL = convertRuleSearchTermToKQL;
exports.convertRuleTagsToKQL = convertRuleTagsToKQL;
exports.convertRuleTypesToKQL = convertRuleTypesToKQL;
exports.convertRulesFilterToKQL = convertRulesFilterToKQL;
var _detection_engine = require("../../api/detection_engine");
var _kql = require("../../utils/kql");
var _rule_fields = require("./rule_fields");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const KQL_FILTER_IMMUTABLE_RULES = exports.KQL_FILTER_IMMUTABLE_RULES = `${_rule_fields.PARAMS_IMMUTABLE_FIELD}: true`;
const KQL_FILTER_MUTABLE_RULES = exports.KQL_FILTER_MUTABLE_RULES = `${_rule_fields.PARAMS_IMMUTABLE_FIELD}: false`;
const KQL_FILTER_ENABLED_RULES = exports.KQL_FILTER_ENABLED_RULES = `${_rule_fields.ENABLED_FIELD}: true`;
const KQL_FILTER_DISABLED_RULES = exports.KQL_FILTER_DISABLED_RULES = `${_rule_fields.ENABLED_FIELD}: false`;
const KQL_FILTER_CUSTOMIZED_RULES = exports.KQL_FILTER_CUSTOMIZED_RULES = `${_rule_fields.IS_CUSTOMIZED_FIELD}: true`;
const KQL_FILTER_NOT_CUSTOMIZED_RULES = exports.KQL_FILTER_NOT_CUSTOMIZED_RULES = `NOT ${_rule_fields.IS_CUSTOMIZED_FIELD}: true`;
/**
 * Convert rules filter options object to KQL query
 *
 * @param filterOptions desired filters (e.g. filter/sortField/sortOrder)
 *
 * @returns KQL string
 */
function convertRulesFilterToKQL({
  filter: searchTerm,
  showCustomRules,
  showElasticRules,
  enabled,
  tags,
  excludeRuleTypes = [],
  ruleExecutionStatus,
  customizationStatus,
  includeRuleTypes = []
}) {
  const kql = [];
  if (searchTerm !== null && searchTerm !== void 0 && searchTerm.trim().length) {
    kql.push(`(${convertRuleSearchTermToKQL(searchTerm)})`);
  }
  if (showCustomRules && showElasticRules) {
    // if both showCustomRules && showElasticRules selected we omit filter, as it includes all existing rules
  } else if (showElasticRules) {
    kql.push(KQL_FILTER_IMMUTABLE_RULES);
  } else if (showCustomRules) {
    kql.push(KQL_FILTER_MUTABLE_RULES);
  }
  if (enabled !== undefined) {
    kql.push(enabled ? KQL_FILTER_ENABLED_RULES : KQL_FILTER_DISABLED_RULES);
  }
  if (tags !== null && tags !== void 0 && tags.length) {
    kql.push(convertRuleTagsToKQL(tags));
  }
  if (excludeRuleTypes.length) {
    kql.push(`NOT ${convertRuleTypesToKQL(excludeRuleTypes)}`);
  }
  if (includeRuleTypes.length) {
    kql.push(convertRuleTypesToKQL(includeRuleTypes));
  }
  if (ruleExecutionStatus === _detection_engine.RuleExecutionStatusEnum.succeeded) {
    kql.push(`${_rule_fields.LAST_RUN_OUTCOME_FIELD}: "succeeded"`);
  } else if (ruleExecutionStatus === _detection_engine.RuleExecutionStatusEnum['partial failure']) {
    kql.push(`${_rule_fields.LAST_RUN_OUTCOME_FIELD}: "warning"`);
  } else if (ruleExecutionStatus === _detection_engine.RuleExecutionStatusEnum.failed) {
    kql.push(`${_rule_fields.LAST_RUN_OUTCOME_FIELD}: "failed"`);
  }
  if (customizationStatus === _detection_engine.RuleCustomizationStatus.CUSTOMIZED) {
    kql.push(KQL_FILTER_CUSTOMIZED_RULES);
  } else if (customizationStatus === _detection_engine.RuleCustomizationStatus.NOT_CUSTOMIZED) {
    kql.push(KQL_FILTER_NOT_CUSTOMIZED_RULES);
  }
  return kql.join(' AND ');
}
const SEARCHABLE_RULE_ATTRIBUTES = [_rule_fields.RULE_PARAMS_FIELDS.INDEX, _rule_fields.RULE_PARAMS_FIELDS.TACTIC_ID, _rule_fields.RULE_PARAMS_FIELDS.TACTIC_NAME, _rule_fields.RULE_PARAMS_FIELDS.TECHNIQUE_ID, _rule_fields.RULE_PARAMS_FIELDS.TECHNIQUE_NAME, _rule_fields.RULE_PARAMS_FIELDS.SUBTECHNIQUE_ID, _rule_fields.RULE_PARAMS_FIELDS.SUBTECHNIQUE_NAME];

/**
 * Build KQL search terms.
 *
 * Note that RULE_NAME_FIELD is special, for single term searches
 * it includes partial matches, supporting special characters.
 *
 * Ie - "sql" =KQL=> *sql* --matches-->  sql, Postgreslq, SQLCMD.EXE
 *    - "sql:" =KQL=> *sql\:* --matches-->  sql:x64, but NOT sql_x64
 *
 * Whereas the rest of the fields, and multiple term searches,
 * we use exact term match with quotations.
 *
 * Ie - "sql" =KQL=> "sql" --matches--> sql server, but NOT mssql or SQLCMD.EXE
 *
 * @param searchTerm search term (ie from the search bar)
 * @returns KQL String
 */
function convertRuleSearchTermToKQL(searchTerm) {
  const searchableConditions = SEARCHABLE_RULE_ATTRIBUTES.map(attribute => `${attribute}: ${(0, _kql.prepareKQLStringParam)(searchTerm)}`);
  const escapedTerm = (0, _kql.fullyEscapeKQLStringParam)(searchTerm);
  const isSingleTerm = escapedTerm.split(' ').length === 1;
  let ruleNameCondition = '';
  if (isSingleTerm) {
    ruleNameCondition = `${_rule_fields.RULE_NAME_FIELD}.keyword: *${escapedTerm}*`;
  } else {
    ruleNameCondition = `${_rule_fields.RULE_NAME_FIELD}: ${(0, _kql.prepareKQLStringParam)(searchTerm)}`;
  }
  return [ruleNameCondition].concat(searchableConditions).join(' OR ');
}
function convertRuleTagsToKQL(tags) {
  return `${_rule_fields.TAGS_FIELD}:(${tags.map(_kql.prepareKQLStringParam).join(' AND ')})`;
}
function convertRuleTypesToKQL(ruleTypes) {
  return `${_rule_fields.PARAMS_TYPE_FIELD}: (${ruleTypes.map(_kql.prepareKQLStringParam).join(' OR ')})`;
}