"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineRoutes = defineRoutes;
var _configSchema = require("@kbn/config-schema");
var _i18n = require("@kbn/i18n");
var _api_routes = require("../common/api_routes");
var _error_handler = require("./utils/error_handler");
var _fetch_synonym_sets = require("./lib/fetch_synonym_sets");
var _pagination = require("../common/pagination");
var _delete_synonyms_set = require("./lib/delete_synonyms_set");
var _fetch_synonyms_set = require("./lib/fetch_synonyms_set");
var _delete_synonym_rule = require("./lib/delete_synonym_rule");
var _fetch_synonym_rule = require("./lib/fetch_synonym_rule");
var _put_synonyms_set = require("./lib/put_synonyms_set");
var _fetch_unique_rule_id = require("./lib/fetch_unique_rule_id");
var _put_synonyms_rule = require("./lib/put_synonyms_rule");
var _fetch_synonyms_set_exists = require("./lib/fetch_synonyms_set_exists");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function defineRoutes({
  logger,
  router
}) {
  router.get({
    path: _api_routes.APIRoutes.SYNONYM_SETS,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_synonyms']
      }
    },
    validate: {
      query: _configSchema.schema.object({
        from: _configSchema.schema.number({
          defaultValue: _pagination.DEFAULT_PAGE_VALUE.from
        }),
        size: _configSchema.schema.number({
          defaultValue: _pagination.DEFAULT_PAGE_VALUE.size
        })
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    const user = core.security.authc.getCurrentUser();
    if (!user) {
      return response.customError({
        statusCode: 502,
        body: 'Could not retrieve current user, security plugin is not ready'
      });
    }
    const hasSearchSynonymsPrivilege = await asCurrentUser.security.hasPrivileges({
      cluster: ['manage_search_synonyms']
    });
    if (!hasSearchSynonymsPrivilege.has_all_requested) {
      return response.forbidden({
        body: "Your user doesn't have manage_search_synonyms privileges"
      });
    }
    const result = await (0, _fetch_synonym_sets.fetchSynonymSets)(asCurrentUser, {
      from: request.query.from,
      size: request.query.size
    });
    return response.ok({
      headers: {
        'content-type': 'application/json'
      },
      body: result
    });
  }));
  router.delete({
    path: _api_routes.APIRoutes.SYNONYM_SET_ID,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_synonyms']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        synonymsSetId: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    const user = core.security.authc.getCurrentUser();
    if (!user) {
      return response.customError({
        statusCode: 502,
        body: 'Could not retrieve current user, security plugin is not ready'
      });
    }
    const synonymsSetId = request.params.synonymsSetId;
    const result = await (0, _delete_synonyms_set.deleteSynonymsSet)(asCurrentUser, synonymsSetId);
    return response.ok({
      headers: {
        'content-type': 'application/json'
      },
      body: result
    });
  }));
  router.get({
    path: _api_routes.APIRoutes.SYNONYM_SET_ID,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_synonyms']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        synonymsSetId: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        from: _configSchema.schema.number({
          defaultValue: _pagination.DEFAULT_PAGE_VALUE.from
        }),
        size: _configSchema.schema.number({
          defaultValue: _pagination.DEFAULT_PAGE_VALUE.size
        })
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    const user = core.security.authc.getCurrentUser();
    if (!user) {
      return response.customError({
        statusCode: 502,
        body: 'Could not retrieve current user, security plugin is not ready'
      });
    }
    const hasSearchSynonymsPrivilege = await asCurrentUser.security.hasPrivileges({
      cluster: ['manage_search_synonyms']
    });
    if (!hasSearchSynonymsPrivilege.has_all_requested) {
      return response.forbidden({
        body: "Your user doesn't have manage_search_synonyms privileges"
      });
    }
    const synonymsSetId = request.params.synonymsSetId;
    const result = await (0, _fetch_synonyms_set.fetchSynonymsSet)(asCurrentUser, synonymsSetId, {
      from: request.query.from,
      size: request.query.size
    });
    return response.ok({
      headers: {
        'content-type': 'application/json'
      },
      body: result
    });
  }));
  router.get({
    path: _api_routes.APIRoutes.SYNONYM_SET_ID_RULE_ID,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_synonyms']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        synonymsSetId: _configSchema.schema.string(),
        ruleId: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    const user = core.security.authc.getCurrentUser();
    if (!user) {
      return response.customError({
        statusCode: 502,
        body: 'Could not retrieve current user, security plugin is not ready'
      });
    }
    const hasSearchSynonymsPrivilege = await asCurrentUser.security.hasPrivileges({
      cluster: ['manage_search_synonyms']
    });
    if (!hasSearchSynonymsPrivilege.has_all_requested) {
      return response.forbidden({
        body: "Your user doesn't have manage_search_synonyms privileges"
      });
    }
    const synonymsSetId = request.params.synonymsSetId;
    const ruleId = request.params.ruleId;
    const result = await (0, _fetch_synonym_rule.fetchSynonymRule)(asCurrentUser, synonymsSetId, ruleId);
    return response.ok({
      headers: {
        'content-type': 'application/json'
      },
      body: result
    });
  }));
  router.delete({
    path: _api_routes.APIRoutes.SYNONYM_SET_ID_RULE_ID,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_synonyms']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        synonymsSetId: _configSchema.schema.string(),
        ruleId: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    const user = core.security.authc.getCurrentUser();
    if (!user) {
      return response.customError({
        statusCode: 502,
        body: 'Could not retrieve current user, security plugin is not ready'
      });
    }
    const hasSearchSynonymsPrivilege = await asCurrentUser.security.hasPrivileges({
      cluster: ['manage_search_synonyms']
    });
    if (!hasSearchSynonymsPrivilege.has_all_requested) {
      return response.forbidden({
        body: "Your user doesn't have manage_search_synonyms privileges"
      });
    }
    const synonymsSetId = request.params.synonymsSetId;
    const ruleId = request.params.ruleId;
    const result = await (0, _delete_synonym_rule.deleteSynonymRule)(asCurrentUser, synonymsSetId, ruleId);
    return response.ok({
      headers: {
        'content-type': 'application/json'
      },
      body: result
    });
  }));
  router.put({
    path: _api_routes.APIRoutes.SYNONYM_SET_ID,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_synonyms']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        synonymsSetId: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        forceWrite: _configSchema.schema.boolean({
          defaultValue: false
        })
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    const user = core.security.authc.getCurrentUser();
    if (!user) {
      return response.customError({
        statusCode: 502,
        body: 'Could not retrieve current user, security plugin is not ready'
      });
    }
    const hasSearchSynonymsPrivilege = await asCurrentUser.security.hasPrivileges({
      cluster: ['manage_search_synonyms']
    });
    if (!hasSearchSynonymsPrivilege.has_all_requested) {
      return response.forbidden({
        body: "Your user doesn't have manage_search_synonyms privileges"
      });
    }
    const synonymsSetId = request.params.synonymsSetId;
    const forceWrite = request.query.forceWrite;
    const isExisting = await (0, _fetch_synonyms_set_exists.fetchSynonymsSetExists)(asCurrentUser, synonymsSetId);
    if (isExisting && !forceWrite) {
      return response.customError({
        statusCode: 409,
        body: _i18n.i18n.translate('xpack.search.synonyms.api.routes.synonymsSetAlreadyExistsErrorMessage', {
          defaultMessage: `Synonyms set {synonymsSetId} already exists. Use forceWrite=true to overwrite it.`,
          values: {
            synonymsSetId
          }
        })
      });
    }
    const result = await (0, _put_synonyms_set.putSynonymsSet)(asCurrentUser, synonymsSetId);
    return response.ok({
      headers: {
        'content-type': 'application/json'
      },
      body: result
    });
  }));
  router.post({
    path: _api_routes.APIRoutes.GENERATE_SYNONYM_RULE_ID,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_synonyms']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        synonymsSetId: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    const user = core.security.authc.getCurrentUser();
    if (!user) {
      return response.customError({
        statusCode: 502,
        body: 'Could not retrieve current user, security plugin is not ready'
      });
    }
    const hasSearchSynonymsPrivilege = await asCurrentUser.security.hasPrivileges({
      cluster: ['manage_search_synonyms']
    });
    if (!hasSearchSynonymsPrivilege.has_all_requested) {
      return response.forbidden({
        body: "Your user doesn't have manage_search_synonyms privileges"
      });
    }
    const synonymsSetId = request.params.synonymsSetId;
    const result = await (0, _fetch_unique_rule_id.fetchUniqueRuleId)(asCurrentUser, synonymsSetId);
    return response.ok({
      headers: {
        'content-type': 'application/json'
      },
      body: {
        id: result
      }
    });
  }));
  router.put({
    path: _api_routes.APIRoutes.SYNONYM_SET_ID_RULE_ID,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: ['manage_synonyms']
      }
    },
    validate: {
      params: _configSchema.schema.object({
        synonymsSetId: _configSchema.schema.string(),
        ruleId: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        synonyms: _configSchema.schema.string()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const core = await context.core;
    const {
      client: {
        asCurrentUser
      }
    } = core.elasticsearch;
    const user = core.security.authc.getCurrentUser();
    if (!user) {
      return response.customError({
        statusCode: 502,
        body: 'Could not retrieve current user, security plugin is not ready'
      });
    }
    const hasSearchSynonymsPrivilege = await asCurrentUser.security.hasPrivileges({
      cluster: ['manage_search_synonyms']
    });
    if (!hasSearchSynonymsPrivilege.has_all_requested) {
      return response.forbidden({
        body: "Your user doesn't have manage_search_synonyms privileges"
      });
    }
    const synonymsSetId = request.params.synonymsSetId;
    const ruleId = request.params.ruleId;
    const synonyms = request.body.synonyms;
    const result = await (0, _put_synonyms_rule.putSynonymsRule)(asCurrentUser, synonymsSetId, ruleId, synonyms);
    return response.ok({
      headers: {
        'content-type': 'application/json'
      },
      body: result
    });
  }));
}