"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSynonymRuleFlyoutState = void 0;
var _i18n = require("@kbn/i18n");
var _react = require("react");
var _synonyms_utils = require("../../utils/synonyms_utils");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useSynonymRuleFlyoutState = ({
  synonymRule,
  flyoutMode,
  renderExplicit = false
}) => {
  const {
    parsedFromTerms,
    parsedToTermsString,
    parsedIsExplicit
  } = (0, _synonyms_utils.synonymToComboBoxOption)(flyoutMode === 'create' ? '' : synonymRule.synonyms);
  const sortedParsedFromTerms = [...parsedFromTerms].sort((a, b) => a.label.localeCompare(b.label));
  const isExplicit = renderExplicit || parsedIsExplicit;
  const [fromTerms, setFromTerms] = (0, _react.useState)(sortedParsedFromTerms);
  const [mapToTerms, setMapToTerms] = (0, _react.useState)(parsedToTermsString);
  const [isFromTermsInvalid, setIsFromTermsInvalid] = (0, _react.useState)(false);
  const [isMapToTermsInvalid, setIsMapToTermsInvalid] = (0, _react.useState)(false);
  const [fromTermErrors, setFromTermErrors] = (0, _react.useState)([]);
  const [mapToTermErrors, setMapToTermErrors] = (0, _react.useState)([]);
  const [currentSortDirection, setCurrentSortDirection] = (0, _react.useState)('ascending');
  const [announcement, setAnnouncement] = (0, _react.useState)('');
  const announce = message => {
    setAnnouncement(message);
    setTimeout(() => setAnnouncement(''), 1000); // clear for next use
  };
  const hasChanges = flyoutMode === 'create' ? (fromTerms.length !== 0 || mapToTerms.length !== 0) && (0, _synonyms_utils.synonymsOptionToString)({
    fromTerms,
    toTerms: mapToTerms,
    isExplicit
  }) !== synonymRule.synonyms : (0, _synonyms_utils.synonymsOptionToString)({
    fromTerms,
    toTerms: mapToTerms,
    isExplicit
  }) !== synonymRule.synonyms;
  const canSave = fromTerms.length > 0 && !(isExplicit && !mapToTerms) && hasChanges && !isFromTermsInvalid && !isMapToTermsInvalid;
  const resetChanges = () => {
    setFromTerms(parsedFromTerms);
    setMapToTerms(parsedToTermsString);
    setIsFromTermsInvalid(false);
    setIsMapToTermsInvalid(false);
    setFromTermErrors([]);
    setMapToTermErrors([]);
    announce(_i18n.i18n.translate('xpack.searchSynonyms.synonymsSetRuleFlyout.resetAnnouncement', {
      defaultMessage: 'All changes have been reset'
    }));
  };
  const isValid = value => {
    const trimmedValue = value.trim();
    if (value !== '' && trimmedValue === '') {
      setIsFromTermsInvalid(true);
      setFromTermErrors([_constants.ERROR_MESSAGES.empty_from_term]);
      return false;
    }
    if (isExplicit && trimmedValue.includes('=>')) {
      setIsFromTermsInvalid(true);
      setFromTermErrors([_constants.ERROR_MESSAGES.multiple_explicit_separator]);
      return false;
    }
    const exists = fromTerms.find(term => term.label === value);
    if (exists !== undefined) {
      setFromTermErrors([_constants.ERROR_MESSAGES.term_exists]);
      setIsFromTermsInvalid(true);
      return false;
    } else {
      setIsFromTermsInvalid(false);
      setFromTermErrors([]);
      return true;
    }
  };
  const isMapToValid = value => {
    const trimmedValue = value.trim();
    if (value !== '' && trimmedValue === '') {
      setIsMapToTermsInvalid(true);
      setMapToTermErrors([_constants.ERROR_MESSAGES.empty_to_term]);
      return false;
    }
    if (trimmedValue.includes('=>')) {
      setIsMapToTermsInvalid(true);
      setMapToTermErrors([_constants.ERROR_MESSAGES.multiple_explicit_separator]);
      return false;
    }
    if (trimmedValue.split(',').some(term => term.trim() === '')) {
      setIsMapToTermsInvalid(true);
      setMapToTermErrors([_constants.ERROR_MESSAGES.empty_to_term]);
      return false;
    }
    setIsMapToTermsInvalid(false);
    return true;
  };
  const onSortTerms = direction => {
    if (!direction) {
      direction = currentSortDirection === 'ascending' ? 'descending' : 'ascending';
    }
    fromTerms.sort((a, b) => direction === 'ascending' ? a.label.localeCompare(b.label) : b.label.localeCompare(a.label));
    setCurrentSortDirection(direction);
    setFromTerms([...fromTerms]);
  };
  const onSearchChange = searchValue => {
    if (!searchValue) {
      setIsFromTermsInvalid(false);
      setFromTermErrors([]);
      return;
    }
    setIsFromTermsInvalid(!isValid(searchValue));
  };
  const onCreateOption = searchValue => {
    if (searchValue.trim() === '') {
      return;
    }
    if (!isValid(searchValue)) {
      return false;
    }
    setFromTerms([...fromTerms, {
      label: searchValue,
      key: searchValue
    }]);
  };
  const removeTermFromOptions = term => {
    setFromTerms(fromTerms.filter(t => t.label !== term.label));
  };
  const clearFromTerms = () => {
    setFromTerms([]);
    announce(_i18n.i18n.translate('xpack.searchSynonyms.synonymsSetRuleFlyout.clearedAnnouncement', {
      defaultMessage: 'All terms have been removed'
    }));
  };
  const onMapToChange = value => {
    isMapToValid(value);
    setMapToTerms(value);
  };
  return {
    canSave,
    clearFromTerms,
    currentSortDirection,
    fromTermErrors,
    fromTerms,
    hasChanges,
    isExplicit,
    isFromTermsInvalid,
    isMapToTermsInvalid,
    mapToTermErrors,
    mapToTerms,
    announcement,
    onCreateOption,
    onMapToChange,
    onSearchChange,
    onSortTerms,
    removeTermFromOptions,
    resetChanges
  };
};
exports.useSynonymRuleFlyoutState = useSynonymRuleFlyoutState;