"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.clearAddClusterErrors = exports.addCluster = void 0;
var _i18n = require("@kbn/i18n");
var _shared_imports = require("../../../shared_imports");
var _services = require("../../services");
var _notification = require("../../services/notification");
var _action_types = require("../action_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const addCluster = cluster => async dispatch => {
  dispatch({
    type: _action_types.ADD_CLUSTER_START
  });
  try {
    await Promise.all([(0, _services.addCluster)(cluster),
    // Wait at least half a second to avoid a weird flicker of the saving feedback.
    new Promise(resolve => setTimeout(resolve, 500))]);
  } catch (error) {
    if (error) {
      const {
        body
      } = error;

      // Expect an error in the shape provided by http service.
      if (body) {
        const {
          statusCode,
          message
        } = body;
        if (statusCode && statusCode === 409) {
          return dispatch({
            type: _action_types.ADD_CLUSTER_FAILURE,
            payload: {
              error: {
                message: _i18n.i18n.translate('xpack.remoteClusters.addAction.clusterNameAlreadyExistsErrorMessage', {
                  defaultMessage: `A cluster with the name ''{clusterName}'' already exists.`,
                  values: {
                    clusterName: cluster.name
                  }
                })
              }
            }
          });
        }
        return dispatch({
          type: _action_types.ADD_CLUSTER_FAILURE,
          payload: {
            error: {
              message: _i18n.i18n.translate('xpack.remoteClusters.addAction.failedDefaultErrorMessage', {
                defaultMessage: 'Request failed with a {statusCode} error. {message}',
                values: {
                  statusCode,
                  message
                }
              })
            }
          }
        });
      }
    }

    // This error isn't an HTTP error, so let the fatal error screen tell the user something
    // unexpected happened.
    return _notification.fatalError.add(error, _i18n.i18n.translate('xpack.remoteClusters.addAction.errorTitle', {
      defaultMessage: 'Error adding cluster'
    }));
  }
  dispatch({
    type: _action_types.ADD_CLUSTER_SUCCESS
  });
  const {
    history,
    route: {
      location: {
        search
      }
    }
  } = (0, _services.getRouter)();
  const {
    redirect: redirectUrl
  } = (0, _shared_imports.extractQueryParams)(search);
  if (redirectUrl) {
    // A toast is only needed if we're leaving the app.
    _notification.toasts.addSuccess(_i18n.i18n.translate('xpack.remoteClusters.addAction.successTitle', {
      defaultMessage: `Added remote cluster ''{name}''`,
      values: {
        name: cluster.name
      }
    }));
    const decodedRedirect = decodeURIComponent(redirectUrl);
    (0, _services.redirect)(`${decodedRedirect}?cluster=${cluster.name}`);
  } else {
    // This will open the new job in the detail panel. Note that we're *not* showing a success toast
    // here, because it would partially obscure the detail panel.
    history.push({
      pathname: `/list`,
      search: `?cluster=${cluster.name}`
    });
  }
};
exports.addCluster = addCluster;
const clearAddClusterErrors = () => dispatch => {
  dispatch({
    type: _action_types.CLEAR_ADD_CLUSTER_ERRORS
  });
};
exports.clearAddClusterErrors = clearAddClusterErrors;