"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerBatchReindexIndicesRoutes = registerBatchReindexIndicesRoutes;
var _configSchema = require("@kbn/config-schema");
var _elasticsearch = require("@elastic/elasticsearch");
var _server = require("@kbn/es-ui-shared-plugin/server");
var _upgradeAssistantPkgServer = require("@kbn/upgrade-assistant-pkg-server");
var _common = require("../../../common");
var _map_any_error_to_kibana_http_response = require("./map_any_error_to_kibana_http_response");
var _reindex_indices = require("./reindex_indices");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerBatchReindexIndicesRoutes({
  router,
  getReindexService
}) {
  // Get the current batch queue
  router.get({
    path: `${_common.REINDEX_SERVICE_BASE_PATH}/batch/queue`,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es client for authorization'
      }
    },
    validate: {}
  }, async ({
    core
  }, request, response) => {
    const {
      elasticsearch: {
        client: esClient
      },
      savedObjects
    } = await core;
    const {
      getClient
    } = savedObjects;
    try {
      const reindexService = (await getReindexService()).getScopedClient({
        savedObjects: getClient({
          includedHiddenTypes: [_upgradeAssistantPkgServer.REINDEX_OP_TYPE]
        }),
        dataClient: esClient,
        request
      });
      const result = await reindexService.getBatchQueueResponse();
      return response.ok({
        body: result
      });
    } catch (error) {
      if (error instanceof _elasticsearch.errors.ResponseError) {
        return (0, _server.handleEsError)({
          error,
          response
        });
      }
      return (0, _map_any_error_to_kibana_http_response.mapAnyErrorToKibanaHttpResponse)(error);
    }
  });

  // Add indices for reindexing to the worker's batch
  router.post({
    path: `${_common.REINDEX_SERVICE_BASE_PATH}/batch`,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es client for authorization'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        indices: _configSchema.schema.arrayOf(_reindex_indices.reindexSchema)
      })
    }
  }, async ({
    core
  }, request, response) => {
    const {
      savedObjects: {
        getClient
      },
      elasticsearch: {
        client: esClient
      }
    } = await core;
    const {
      indices
    } = request.body;
    const reindexService = (await getReindexService()).getScopedClient({
      savedObjects: getClient({
        includedHiddenTypes: [_upgradeAssistantPkgServer.REINDEX_OP_TYPE]
      }),
      dataClient: esClient,
      request
    });
    const results = await reindexService.addToBatch(indices);
    return response.ok({
      body: results
    });
  });
}