"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerStorageExplorerRoute = registerStorageExplorerRoute;
var _configSchema = require("@kbn/config-schema");
var _lodash = require("lodash");
var _common = require("../../../common");
var _storage_explorer = require("../../../common/storage_explorer");
var _compat = require("../compat");
var _get_daily_data_generation = require("./get_daily_data_generation.size");
var _get_host_breakdown_size_timeseries = require("./get_host_breakdown_size_timeseries");
var _get_host_details = require("./get_host_details");
var _get_host_distinct_probabilistic_count = require("./get_host_distinct_probabilistic_count");
var _get_indices_stats = require("./get_indices_stats");
var _get_storage_details_grouped_by_index = require("./get_storage_details_grouped_by_index");
var _get_storage_details_per_index = require("./get_storage_details_per_index");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerStorageExplorerRoute({
  router,
  services: {
    createProfilingEsClient
  }
}) {
  const paths = (0, _common.getRoutePaths)();
  router.get({
    path: paths.StorageExplorerSummary,
    security: {
      authz: {
        requiredPrivileges: ['profiling']
      }
    },
    validate: {
      query: _configSchema.schema.object({
        indexLifecyclePhase: _configSchema.schema.oneOf([_configSchema.schema.literal(_storage_explorer.IndexLifecyclePhaseSelectOption.All), _configSchema.schema.literal(_storage_explorer.IndexLifecyclePhaseSelectOption.Hot), _configSchema.schema.literal(_storage_explorer.IndexLifecyclePhaseSelectOption.Warm), _configSchema.schema.literal(_storage_explorer.IndexLifecyclePhaseSelectOption.Cold), _configSchema.schema.literal(_storage_explorer.IndexLifecyclePhaseSelectOption.Frozen)]),
        timeFrom: _configSchema.schema.number(),
        timeTo: _configSchema.schema.number(),
        kuery: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    var _totalIndicesStats$_a, _totalIndicesStats$_a2, _totalIndicesStats$_a3, _totalSymbolsIndicesS, _totalSymbolsIndicesS2, _totalSymbolsIndicesS3;
    const {
      timeFrom,
      timeTo,
      kuery,
      indexLifecyclePhase
    } = request.query;
    const client = await (0, _compat.getClient)(context);
    const profilingClient = createProfilingEsClient({
      request,
      esClient: client
    });
    const profilingEsClient = profilingClient.getEsClient();
    const [totalIndicesStats, totalSymbolsIndicesStats, nodeStats, hostAndDistinctProbabilisticCount] = await Promise.all([(0, _get_indices_stats.getIndicesStats)({
      client: profilingEsClient,
      indices: _get_indices_stats.allIndices
    }), (0, _get_indices_stats.getIndicesStats)({
      client: profilingEsClient,
      indices: _get_indices_stats.symbolsIndices
    }), (0, _get_indices_stats.getNodesStats)({
      client: profilingEsClient
    }), (0, _get_host_distinct_probabilistic_count.getHostAndDistinctProbabilisticCount)({
      client: profilingClient,
      timeFrom,
      timeTo,
      kuery,
      indexLifecyclePhase
    })]);
    const {
      dailyDataGenerationBytes
    } = await (0, _get_daily_data_generation.getDailyDataGenerationSize)({
      client: profilingClient,
      timeFrom,
      timeTo,
      allIndicesStats: totalIndicesStats.indices,
      kuery
    });
    const {
      nodes: diskSpacePerNode
    } = nodeStats;
    const {
      totalNumberOfDistinctProbabilisticValues,
      totalNumberOfHosts
    } = hostAndDistinctProbabilisticCount;
    const totalProfilingSizeBytes = (_totalIndicesStats$_a = (_totalIndicesStats$_a2 = totalIndicesStats._all.total) === null || _totalIndicesStats$_a2 === void 0 ? void 0 : (_totalIndicesStats$_a3 = _totalIndicesStats$_a2.store) === null || _totalIndicesStats$_a3 === void 0 ? void 0 : _totalIndicesStats$_a3.size_in_bytes) !== null && _totalIndicesStats$_a !== void 0 ? _totalIndicesStats$_a : 0;
    const totalSymbolsSizeBytes = (_totalSymbolsIndicesS = (_totalSymbolsIndicesS2 = totalSymbolsIndicesStats._all.total) === null || _totalSymbolsIndicesS2 === void 0 ? void 0 : (_totalSymbolsIndicesS3 = _totalSymbolsIndicesS2.store) === null || _totalSymbolsIndicesS3 === void 0 ? void 0 : _totalSymbolsIndicesS3.size_in_bytes) !== null && _totalSymbolsIndicesS !== void 0 ? _totalSymbolsIndicesS : 0;
    const totalDiskSpace = (0, _lodash.sumBy)((0, _lodash.values)(diskSpacePerNode), node => {
      var _node$fs$total$total_, _node$fs, _node$fs$total;
      return (_node$fs$total$total_ = node === null || node === void 0 ? void 0 : (_node$fs = node.fs) === null || _node$fs === void 0 ? void 0 : (_node$fs$total = _node$fs.total) === null || _node$fs$total === void 0 ? void 0 : _node$fs$total.total_in_bytes) !== null && _node$fs$total$total_ !== void 0 ? _node$fs$total$total_ : 0;
    });
    const summary = {
      totalProfilingSizeBytes,
      totalSymbolsSizeBytes,
      diskSpaceUsedPct: totalProfilingSizeBytes / totalDiskSpace,
      totalNumberOfDistinctProbabilisticValues,
      totalNumberOfHosts,
      dailyDataGenerationBytes
    };
    return response.ok({
      body: summary
    });
  });
  router.get({
    path: paths.StorageExplorerHostStorageDetails,
    security: {
      authz: {
        requiredPrivileges: ['profiling']
      }
    },
    validate: {
      query: _configSchema.schema.object({
        indexLifecyclePhase: _configSchema.schema.oneOf([_configSchema.schema.literal(_storage_explorer.IndexLifecyclePhaseSelectOption.All), _configSchema.schema.literal(_storage_explorer.IndexLifecyclePhaseSelectOption.Hot), _configSchema.schema.literal(_storage_explorer.IndexLifecyclePhaseSelectOption.Warm), _configSchema.schema.literal(_storage_explorer.IndexLifecyclePhaseSelectOption.Cold), _configSchema.schema.literal(_storage_explorer.IndexLifecyclePhaseSelectOption.Frozen)]),
        timeFrom: _configSchema.schema.number(),
        timeTo: _configSchema.schema.number(),
        kuery: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const client = await (0, _compat.getClient)(context);
    const profilingClient = createProfilingEsClient({
      request,
      esClient: client
    });
    const {
      timeFrom,
      timeTo,
      kuery,
      indexLifecyclePhase
    } = request.query;
    const [hostDetailsTimeseries, hostDetails] = await Promise.all([(0, _get_host_breakdown_size_timeseries.getHostBreakdownSizeTimeseries)({
      client: profilingClient,
      timeFrom,
      timeTo,
      kuery,
      indexLifecyclePhase
    }), (0, _get_host_details.getHostDetails)({
      client: profilingClient,
      timeFrom,
      timeTo,
      kuery,
      indexLifecyclePhase
    })]);
    return response.ok({
      body: {
        hostDetailsTimeseries,
        hostDetails
      }
    });
  });
  router.get({
    path: paths.StorageExplorerIndicesStorageDetails,
    security: {
      authz: {
        requiredPrivileges: ['profiling']
      }
    },
    validate: {
      query: _configSchema.schema.object({
        indexLifecyclePhase: _configSchema.schema.oneOf([_configSchema.schema.literal(_storage_explorer.IndexLifecyclePhaseSelectOption.All), _configSchema.schema.literal(_storage_explorer.IndexLifecyclePhaseSelectOption.Hot), _configSchema.schema.literal(_storage_explorer.IndexLifecyclePhaseSelectOption.Warm), _configSchema.schema.literal(_storage_explorer.IndexLifecyclePhaseSelectOption.Cold), _configSchema.schema.literal(_storage_explorer.IndexLifecyclePhaseSelectOption.Frozen)])
      })
    }
  }, async (context, request, response) => {
    const client = await (0, _compat.getClient)(context);
    const profilingClient = createProfilingEsClient({
      request,
      esClient: client
    });
    const profilingEsClient = profilingClient.getEsClient();
    const {
      indexLifecyclePhase
    } = request.query;
    const [storageDetailsGroupedByIndex, storageDetailsPerIndex] = await Promise.all([(0, _get_storage_details_grouped_by_index.getStorageDetailsGroupedByIndex)({
      client: profilingEsClient,
      indexLifecyclePhase
    }), (0, _get_storage_details_per_index.getStorageDetailsPerIndex)({
      client: profilingEsClient,
      indexLifecyclePhase
    })]);
    return response.ok({
      body: {
        storageDetailsGroupedByIndex,
        storageDetailsPerIndex
      }
    });
  });
}