"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerFlameChartSearchRoute = registerFlameChartSearchRoute;
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/observability-plugin/server");
var _ = require(".");
var _common = require("../../common");
var _handle_route_error_handler = require("../utils/handle_route_error_handler");
var _compat = require("./compat");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerFlameChartSearchRoute({
  router,
  logger,
  dependencies: {
    start: {
      profilingDataAccess
    }
  }
}) {
  const paths = (0, _common.getRoutePaths)();
  router.get({
    path: paths.Flamechart,
    security: {
      authz: {
        requiredPrivileges: ['profiling']
      }
    },
    options: {
      timeout: {
        idleSocket: _.IDLE_SOCKET_TIMEOUT
      }
    },
    validate: {
      query: _configSchema.schema.object({
        timeFrom: _configSchema.schema.number(),
        timeTo: _configSchema.schema.number(),
        kuery: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const {
      timeFrom,
      timeTo,
      kuery
    } = request.query;
    const core = await context.core;
    const startSecs = timeFrom / 1000;
    const endSecs = timeTo / 1000;
    try {
      const esClient = await (0, _compat.getClient)(context);
      const flamegraph = await profilingDataAccess.services.fetchFlamechartData({
        core,
        esClient,
        totalSeconds: endSecs - startSecs,
        query: {
          bool: {
            filter: [...(0, _server.kqlQuery)(kuery), {
              range: {
                ['@timestamp']: {
                  gte: String(startSecs),
                  lt: String(endSecs),
                  format: 'epoch_second'
                }
              }
            }]
          }
        }
      });
      return response.ok({
        body: flamegraph
      });
    } catch (error) {
      return (0, _handle_route_error_handler.handleRouteHandlerError)({
        error,
        logger,
        response,
        message: 'Error while fetching flamegraph'
      });
    }
  });
}