"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getPresentationLabsService = exports.applyProjectStatus = void 0;
var _common = require("../../common");
var _kibana_services = require("./kibana_services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getPresentationLabsService = () => {
  const {
    uiSettings
  } = _kibana_services.coreServices;
  const localStorage = window.localStorage;
  const sessionStorage = window.sessionStorage;
  const getProjects = (solutions = []) => _common.projectIDs.reduce((acc, id) => {
    const project = getProject(id);
    if (solutions.length === 0 || solutions.some(solution => project.solutions.includes(solution))) {
      acc[id] = project;
    }
    return acc;
  }, {});
  const getProject = id => {
    const project = _common.projects[id];
    const status = {
      session: isEnabledByStorageValue(project, 'session', sessionStorage.getItem(id)),
      browser: isEnabledByStorageValue(project, 'browser', localStorage.getItem(id)),
      kibana: isEnabledByStorageValue(project, 'kibana', uiSettings.get(id, project.isActive))
    };
    return applyProjectStatus(project, status);
  };
  const setProjectStatus = (name, env, status) => {
    switch (env) {
      case 'session':
        setStorageStatus(sessionStorage, name, status);
        break;
      case 'browser':
        setStorageStatus(localStorage, name, status);
        break;
      case 'kibana':
        setUISettingsStatus(uiSettings, name, status);
        break;
    }
  };
  const reset = () => {
    clearLabsFromStorage(localStorage);
    clearLabsFromStorage(sessionStorage);
    _common.projectIDs.forEach(id => setProjectStatus(id, 'kibana', _common.projects[id].isActive));
  };
  const isProjectEnabled = id => getProject(id).status.isEnabled;
  return {
    getProjects,
    getProject,
    isProjectEnabled,
    reset,
    setProjectStatus
  };
};

/**
 * Helpers
 */
exports.getPresentationLabsService = getPresentationLabsService;
const isEnabledByStorageValue = (project, environment, value) => {
  const defaultValue = project.isActive;
  if (!project.environments.includes(environment)) {
    return defaultValue;
  }
  if (value === true || value === false) {
    return value;
  }
  if (value === 'enabled') {
    return true;
  }
  if (value === 'disabled') {
    return false;
  }
  return defaultValue;
};
const setStorageStatus = (storage, id, enabled) => storage.setItem(id, enabled ? 'enabled' : 'disabled');
const applyProjectStatus = (project, status) => {
  const {
    isActive,
    environments
  } = project;
  environments.forEach(name => {
    if (!environments.includes(name)) {
      delete status[name];
    }
  });
  const isEnabled = (0, _common.isProjectEnabledByStatus)(isActive, status);
  const isOverride = isEnabled !== isActive;
  return {
    ...project,
    status: {
      ...status,
      defaultValue: isActive,
      isEnabled,
      isOverride
    }
  };
};
exports.applyProjectStatus = applyProjectStatus;
const setUISettingsStatus = (client, id, enabled) => client.set(id, enabled);
const clearLabsFromStorage = storage => {
  _common.projectIDs.forEach(projectID => storage.removeItem(projectID));

  // This is a redundancy, to catch any labs that may have been removed above.
  // We could consider gathering telemetry to see how often this happens, or this may be unnecessary.
  Object.keys(storage).filter(key => key.startsWith(_common.LABS_PROJECT_PREFIX)).forEach(key => storage.removeItem(key));
};