"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KibanaAgentExecutor = void 0;
var _onechatCommon = require("@kbn/onechat-common");
var _rxjs = require("rxjs");
var _uuid = require("uuid");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const generateMessageId = () => `msg-${(0, _uuid.v4)()}`;
const A2A_CONVERSATION_ID_PREFIX = 'a2a-';
const generateA2AConversationId = id => `${A2A_CONVERSATION_ID_PREFIX}${id}`;

/**
 * Agent executor that bridges A2A requests to Kibana's onechat system
 */
class KibanaAgentExecutor {
  constructor(logger, getInternalServices, kibanaRequest, agentId) {
    this.logger = logger;
    this.getInternalServices = getInternalServices;
    this.kibanaRequest = kibanaRequest;
    this.agentId = agentId;
  }
  async execute(requestContext, eventBus) {
    const {
      taskId,
      userMessage,
      contextId
    } = requestContext;
    try {
      this.logger.debug(`A2A: Starting task ${taskId} with contextId ${contextId}`);
      const userText = userMessage.parts.filter(part => part.kind === 'text').map(part => part.text).join(' ');
      const {
        chat
      } = this.getInternalServices();
      const a2aConversationId = generateA2AConversationId(contextId);
      const chatEvents$ = chat.converse({
        agentId: this.agentId,
        nextInput: {
          message: userText
        },
        request: this.kibanaRequest,
        conversationId: a2aConversationId,
        capabilities: {
          visualizations: false
        },
        autoCreateConversationWithId: true
      });

      // Process chat response
      const events = await (0, _rxjs.firstValueFrom)(chatEvents$.pipe((0, _rxjs.toArray)()));
      const roundCompleteEvent = events.find(_onechatCommon.isRoundCompleteEvent);
      if (!roundCompleteEvent) {
        throw new Error('No complete response received from chat service');
      }
      const responseText = roundCompleteEvent.data.round.response.message;

      // Send response back through A2A
      eventBus.publish({
        kind: 'message',
        role: 'agent',
        messageId: generateMessageId(),
        parts: [{
          kind: 'text',
          text: responseText
        }],
        taskId,
        contextId: requestContext.contextId
      });
      eventBus.finished();
      this.logger.debug(`A2A: Task ${taskId} completed`);
    } catch (error) {
      this.logger.error(`A2A: Task ${taskId} failed: ${error}`);
      this.sendErrorResponse(eventBus, taskId, requestContext.contextId, error);
    }
  }
  async cancelTask(taskId, eventBus) {
    this.logger.debug(`A2A: Canceling task ${taskId}`);
    eventBus.publish({
      kind: 'message',
      role: 'agent',
      messageId: generateMessageId(),
      parts: [{
        kind: 'text',
        text: 'Task was canceled.'
      }],
      taskId,
      contextId: taskId
    });
    eventBus.finished();
  }
  sendErrorResponse(eventBus, taskId, contextId, error) {
    const errorMessage = error instanceof Error ? error.message : String(error);
    eventBus.publish({
      kind: 'message',
      role: 'agent',
      messageId: generateMessageId(),
      parts: [{
        kind: 'text',
        text: `Error: ${errorMessage}`
      }],
      taskId,
      contextId
    });
    eventBus.finished();
  }
}
exports.KibanaAgentExecutor = KibanaAgentExecutor;