"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getIndexSearchToolType = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _onechatGenaiUtils = require("@kbn/onechat-genai-utils");
var _validate_configuration = require("./validate_configuration");
var _schemas = require("./schemas");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const searchSchema = _zod.z.object({
  nlQuery: _zod.z.string().describe('A natural language query expressing the search request')
});
const getIndexSearchToolType = () => {
  return {
    toolType: _onechatCommon.ToolType.index_search,
    getDynamicProps: config => {
      return {
        getHandler: () => {
          return async ({
            nlQuery
          }, {
            esClient,
            modelProvider,
            logger,
            events
          }) => {
            const {
              pattern
            } = config;
            const results = await (0, _onechatGenaiUtils.runSearchTool)({
              nlQuery,
              index: pattern,
              esClient: esClient.asCurrentUser,
              model: await modelProvider.getDefaultModel(),
              events,
              logger
            });
            return {
              results
            };
          };
        },
        getSchema: () => searchSchema,
        getLlmDescription: opts => {
          return getFullDescription({
            description: opts.description,
            pattern: opts.config.pattern
          });
        }
      };
    },
    createSchema: _schemas.configurationSchema,
    updateSchema: _schemas.configurationUpdateSchema,
    validateForCreate: async ({
      config,
      context: {
        esClient
      }
    }) => {
      await (0, _validate_configuration.validateConfig)({
        config,
        esClient
      });
      return config;
    },
    validateForUpdate: async ({
      update,
      current,
      context: {
        esClient
      }
    }) => {
      const mergedConfig = {
        ...current,
        ...update
      };
      await (0, _validate_configuration.validateConfig)({
        config: mergedConfig,
        esClient
      });
      return mergedConfig;
    }
  };
};
exports.getIndexSearchToolType = getIndexSearchToolType;
const getFullDescription = ({
  pattern,
  description
}) => {
  return `${description}
  ## Tool usage

  This tool is a a powerful search tool for searching and analyzing data within your Elasticsearch cluster.
  It is configured to search against the following index pattern: \`${pattern}\`.
  It supports both full-text relevance searches and structured analytical queries, based on a natural language query.

  Examples of queries:
  - "find documents about serverless architecture"
  - "search for documents mentioning '[some term]' or '[another term]'"
  - "list all documents where the category is 'electronics'"
  - "show me the last 5 documents from that index"
  - "show me the sales over the last year break down by month"
`;
};