"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateConversation$ = exports.getConversation$ = exports.createPlaceholderConversation$ = exports.createConversation$ = exports.conversationExists$ = void 0;
var _uuid = require("uuid");
var _rxjs = require("rxjs");
var _events = require("./events");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Persist a new conversation and emit the corresponding event
 */
const createConversation$ = ({
  agentId,
  conversationClient,
  conversationId,
  title$,
  roundCompletedEvents$
}) => {
  return (0, _rxjs.forkJoin)({
    title: title$,
    roundCompletedEvent: roundCompletedEvents$
  }).pipe((0, _rxjs.switchMap)(({
    title,
    roundCompletedEvent
  }) => {
    return conversationClient.create({
      id: conversationId,
      title,
      agent_id: agentId,
      rounds: [roundCompletedEvent.data.round]
    });
  }), (0, _rxjs.switchMap)(createdConversation => {
    return (0, _rxjs.of)((0, _events.createConversationCreatedEvent)(createdConversation));
  }));
};

/**
 * Update an existing conversation and emit the corresponding event
 */
exports.createConversation$ = createConversation$;
const updateConversation$ = ({
  conversationClient,
  conversation$,
  title$,
  roundCompletedEvents$
}) => {
  return (0, _rxjs.forkJoin)({
    conversation: conversation$,
    title: title$,
    roundCompletedEvent: roundCompletedEvents$
  }).pipe((0, _rxjs.switchMap)(({
    conversation,
    title,
    roundCompletedEvent
  }) => {
    return conversationClient.update({
      id: conversation.id,
      title,
      rounds: [...conversation.rounds, roundCompletedEvent.data.round]
    });
  }), (0, _rxjs.switchMap)(updatedConversation => {
    return (0, _rxjs.of)((0, _events.createConversationUpdatedEvent)(updatedConversation));
  }));
};
exports.updateConversation$ = updateConversation$;
const conversationExists$ = ({
  conversationId,
  conversationClient
}) => {
  return (0, _rxjs.defer)(() => conversationClient.exists(conversationId));
};
exports.conversationExists$ = conversationExists$;
const getConversation$ = ({
  agentId,
  conversationId,
  autoCreateConversationWithId = false,
  conversationClient
}) => {
  return (0, _rxjs.defer)(() => {
    if (conversationId) {
      if (autoCreateConversationWithId) {
        return conversationExists$({
          conversationId,
          conversationClient
        }).pipe((0, _rxjs.switchMap)(exists => {
          if (exists) {
            return conversationClient.get(conversationId);
          } else {
            return (0, _rxjs.of)(placeholderConversation({
              conversationId,
              agentId
            }));
          }
        }));
      } else {
        return conversationClient.get(conversationId);
      }
    } else {
      return (0, _rxjs.of)(placeholderConversation({
        agentId
      }));
    }
  }).pipe((0, _rxjs.shareReplay)());
};
exports.getConversation$ = getConversation$;
const createPlaceholderConversation$ = ({
  agentId,
  conversationId
}) => {
  return (0, _rxjs.of)(placeholderConversation({
    agentId,
    conversationId
  }));
};
exports.createPlaceholderConversation$ = createPlaceholderConversation$;
const placeholderConversation = ({
  agentId,
  conversationId
}) => {
  return {
    id: conversationId !== null && conversationId !== void 0 ? conversationId : (0, _uuid.v4)(),
    title: 'New conversation',
    agent_id: agentId,
    rounds: [],
    updated_at: new Date().toISOString(),
    created_at: new Date().toISOString(),
    user: {
      id: 'unknown',
      username: 'unknown'
    }
  };
};