"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSearchToolGraph = void 0;
var _langgraph = require("@langchain/langgraph");
var _messages = require("@langchain/core/messages");
var _prebuilt = require("@langchain/langgraph/prebuilt");
var _onechatServer = require("@kbn/onechat-server");
var _langchain = require("../../langchain");
var _index_explorer = require("../index_explorer");
var _inner_tools = require("./inner_tools");
var _prompts = require("./prompts");
var _i18n = require("./i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const StateAnnotation = _langgraph.Annotation.Root({
  // inputs
  nlQuery: (0, _langgraph.Annotation)(),
  targetPattern: (0, _langgraph.Annotation)(),
  // inner
  indexIsValid: (0, _langgraph.Annotation)(),
  searchTarget: (0, _langgraph.Annotation)(),
  messages: (0, _langgraph.Annotation)({
    reducer: _langgraph.messagesStateReducer,
    default: () => []
  }),
  // outputs
  error: (0, _langgraph.Annotation)(),
  results: (0, _langgraph.Annotation)({
    reducer: (a, b) => [...a, ...b],
    default: () => []
  })
});
const createSearchToolGraph = ({
  model,
  esClient,
  logger,
  events
}) => {
  const tools = [(0, _inner_tools.createRelevanceSearchTool)({
    model,
    esClient,
    events
  }), (0, _inner_tools.createNaturalLanguageSearchTool)({
    model,
    esClient,
    events,
    logger
  })];
  const toolNode = new _prebuilt.ToolNode(tools);
  const selectAndValidateIndex = async state => {
    var _state$targetPattern;
    events === null || events === void 0 ? void 0 : events.reportProgress(_i18n.progressMessages.selectingTarget());
    const explorerRes = await (0, _index_explorer.indexExplorer)({
      nlQuery: state.nlQuery,
      indexPattern: (_state$targetPattern = state.targetPattern) !== null && _state$targetPattern !== void 0 ? _state$targetPattern : '*',
      esClient,
      model,
      logger,
      limit: 1
    });
    if (explorerRes.resources.length > 0) {
      const selectedResource = explorerRes.resources[0];
      events === null || events === void 0 ? void 0 : events.reportProgress(_i18n.progressMessages.selectedTarget(selectedResource.name));
      return {
        indexIsValid: true,
        searchTarget: {
          type: selectedResource.type,
          name: selectedResource.name
        }
      };
    } else {
      return {
        indexIsValid: false,
        error: `Could not figure out which index to use`
      };
    }
  };
  const terminateIfInvalidIndex = async state => {
    return state.indexIsValid ? 'agent' : '__end__';
  };
  const searchModel = model.chatModel.bindTools(tools).withConfig({
    tags: ['onechat-search-tool']
  });
  const callSearchAgent = async state => {
    events === null || events === void 0 ? void 0 : events.reportProgress(_i18n.progressMessages.resolvingSearchStrategy());
    const response = await searchModel.invoke((0, _prompts.getSearchPrompt)({
      nlQuery: state.nlQuery,
      searchTarget: state.searchTarget
    }));
    return {
      messages: [response]
    };
  };
  const decideContinueOrEnd = async state => {
    // only one call for now
    return '__end__';
  };
  const executeTool = async state => {
    const toolNodeResult = await toolNode.invoke(state.messages);
    const toolResults = extractToolResults(toolNodeResult[toolNodeResult.length - 1]);
    return {
      messages: [...toolNodeResult],
      results: [...toolResults]
    };
  };
  const graph = new _langgraph.StateGraph(StateAnnotation)
  // nodes
  .addNode('check_index', selectAndValidateIndex).addNode('agent', callSearchAgent).addNode('execute_tool', executeTool)
  // edges
  .addEdge('__start__', 'check_index').addConditionalEdges('check_index', terminateIfInvalidIndex, {
    agent: 'agent',
    __end__: '__end__'
  }).addEdge('agent', 'execute_tool').addConditionalEdges('execute_tool', decideContinueOrEnd, {
    agent: 'agent',
    __end__: '__end__'
  }).compile();
  return graph;
};
exports.createSearchToolGraph = createSearchToolGraph;
const extractToolResults = message => {
  if (!(0, _messages.isToolMessage)(message)) {
    throw new Error(`Trying to extract tool results for non-tool result`);
  }
  if (message.artifact) {
    if (!Array.isArray(message.artifact.results)) {
      throw new Error(`Artifact is not a structured tool artifact. Received artifact=${JSON.stringify(message.artifact)}`);
    }
    return message.artifact.results;
  } else {
    const content = (0, _langchain.extractTextContent)(message);
    if (content.startsWith('Error:')) {
      return [(0, _onechatServer.createErrorResult)(content)];
    } else {
      throw new Error(`No artifact attached to tool message. Content was ${message.content}`);
    }
  }
};